import { DISMISS_REASON, IModalProps, ModalLayout } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useMemo, useState, useEffect } from 'react';
import { useDispatch } from 'react-redux';
import { Formik } from 'formik';
import { Button } from '@yandex-cloud/uikit';
import { doDeployTicketAction, useNetworkErrors, useNetworkRequests, useRequestWithLimit } from '../../../redux';
import { DeployTicketActionStatus } from './components/DeployTicketActionStatus/DeployTicketActionStatus';
import { getMassActionRequestKey } from './model';
import {
   DeployTicketActionForm,
   deployTicketFormInitialValue,
   TicketActionParams,
   ticketActionParamsValidationSchema,
} from '../DeployTicketActionForm/DeployTicketActionForm';

import modalClasses from '../../../design/commonModals.module.css';
import classes from './DeployTicketMassActionModal.module.css';
import { RequestMap } from '../../network';
import { networkSlice } from '../../../redux/slices/network/network';
import { useDeployTicketGroups } from '../hooks/useDeployTicketGroups';
import { DeployTicketSafeModePlate } from './components/DeployTicketSafeModePlate/DeployTicketSafeModePlate';
import { TicketAction } from '../../../models/ui';

const DEFAULT_LIMIT = 10;
const SAFE_LIMIT = 1;

/**
 * `ticketId -> Set<patchId>`
 */
type TicketIdsMap = Map<string, Set<string>>;

interface Props extends IModalProps<TicketActionParams> {
   action: TicketAction;
   tickets: TicketIdsMap;
   onStart(): void;
   initialMessage: string;
}

type DoDeployTicketAction = typeof doDeployTicketAction;

export const DeployTicketMassActionModal: React.FC<Props> = React.memo(
   ({ tickets, action, initialMessage, cancel, onStart }) => {
      const ids = useMemo(() => [...tickets.keys()], [tickets]);
      const idsSet = useMemo(() => new Set(ids), [ids]);
      const idsOrder = useMemo(() => new Map(ids.map((id, i) => [id, i])), [ids]);

      const ticketGroups = useDeployTicketGroups(idsSet);

      const [requestStart, setRequestStart] = useState(false);

      const getKey = useMemo(() => getMassActionRequestKey(action), [action]);
      const requestKeys = useMemo(() => ids.map(id => getKey(id)), [getKey, ids]);

      const actions: (message: string, reason: string) => ReturnType<DoDeployTicketAction>[] = useCallback(
         (message, reason) =>
            ids.map((id, i) =>
               doDeployTicketAction.withRequestKey(requestKeys[i])(
                  action,
                  id,
                  [...(tickets.get(id) ?? [])],
                  message,
                  reason,
               ),
            ),
         [action, ids, requestKeys, tickets],
      );

      const dismiss = useCallback(() => {
         cancel(DISMISS_REASON);
      }, [cancel]);

      const dispatch = useDispatch();

      const [requestActions, setRequestActions] = useState<ReturnType<DoDeployTicketAction>[]>([]);
      const items = useMemo(
         () => requestActions.map((requestAction, i) => ({ key: requestKeys[i], action: requestAction })),
         [requestActions, requestKeys],
      );

      const safeMode =
         action === TicketAction.Commit &&
         (ticketGroups.stagesWithMultiTickets.size > 0 || ticketGroups.unknownTicketIds.size > 0);

      const limit = safeMode ? SAFE_LIMIT : DEFAULT_LIMIT;

      const { start } = useRequestWithLimit(items, limit);

      // массовые запросы
      const doRequest = useCallback(
         (message: string, reason: string) => {
            setRequestStart(true);
            dispatch(networkSlice.actions.remove({ requestKeys }));
            setRequestActions(actions(message, reason));
            onStart();
         },
         [actions, dispatch, onStart, requestKeys],
      );

      useEffect(() => {
         if (requestStart) {
            start();
         }
      }, [requestStart, start]);

      const handleSubmit = useCallback(
         (values: TicketActionParams) => {
            doRequest(values.message, values.reason);
         },
         [doRequest],
      );

      const requests = useNetworkRequests(requestKeys);
      const errors = useNetworkErrors(requestKeys);

      const statuses = useMemo(
         () =>
            requestKeys
               .map(key => (requestStart ? requests[key]?.state : null))
               .map(state => ({ requestState: state ?? null })),
         [requestKeys, requestStart, requests],
      );

      const initialValuePatched = useMemo(() => ({ ...deployTicketFormInitialValue, message: initialMessage }), [
         initialMessage,
      ]);

      const status = useCallback(
         id => (
            <DeployTicketActionStatus
               key={id}
               action={action}
               status={requestStart ? requests[requestKeys[idsOrder.get(id)!]]?.state ?? null : null}
               error={requestStart ? errors[requestKeys[idsOrder.get(id)!]] ?? null : null}
               ticketId={id}
               patchIds={tickets.get(id)!}
            />
         ),
         [action, errors, idsOrder, requestKeys, requestStart, requests, tickets],
      );

      return (
         <ModalLayout title={<h2>Deploy ticket mass action: {action}</h2>} onDismiss={dismiss} showFooter={false}>
            <div className={classes.map}>
               <RequestMap statuses={statuses} showCount={true} />
            </div>
            <div className={classes.statuses}>
               {Object.keys(ticketGroups.ticketsByStage).map(stageId => (
                  <>
                     <div>
                        <em>{stageId}</em>
                     </div>
                     {Array.from(ticketGroups.ticketsByStage[stageId].ticketIds.keys()).map(status)}
                  </>
               ))}
               {ticketGroups.unknownTicketIds.size > 0 && (
                  <>
                     <div>Unknown stages</div>
                     {Array.from(ticketGroups.unknownTicketIds).map(status)}
                  </>
               )}
            </div>
            {safeMode && (
               <div className={classes.warning}>
                  <DeployTicketSafeModePlate deployTicketGroups={ticketGroups} />
               </div>
            )}
            <Formik
               initialValues={initialValuePatched}
               onSubmit={handleSubmit}
               validationSchema={ticketActionParamsValidationSchema}
            >
               {form => (
                  <DeployTicketActionForm action={action}>
                     <footer className={modalClasses.footer}>
                        <Button
                           size={'m'}
                           view={'flat'}
                           onClick={dismiss}
                           disabled={form.isSubmitting}
                           className={modalClasses.button}
                        >
                           Cancel
                        </Button>

                        <Button
                           size={'m'}
                           view={'action'}
                           type={'submit'}
                           className={modalClasses.button}
                           disabled={!form.isValid || form.isSubmitting}
                        >
                           Start
                        </Button>
                     </footer>
                  </DeployTicketActionForm>
               )}
            </Formik>
         </ModalLayout>
      );
   },
);

DeployTicketMassActionModal.displayName = 'DeployTicketMassActionModal';
