import React, { ReactNode, useCallback, useMemo } from 'react';

import { Spin } from 'lego-on-react';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faCheck } from '@fortawesome/pro-regular-svg-icons';
import { faExclamationTriangle } from '@fortawesome/pro-solid-svg-icons';
import { modalService } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-cloud/uikit';

import { useDeployTicket } from '../../../../../redux';
import { NetworkItem, RequestState } from '../../../../../redux/slices/network/model';
import { YpErrorModal } from '../../../../network';

import classes from './DeployTicketActionStatus.module.css';
import { TDeployTicketSpec_EDeployTicketSourceType } from '../../../../../proto-typings';
import { TicketAction } from '../../../../../models/ui';

interface Props {
   action: TicketAction;
   ticketId: string;
   patchIds: Set<string>;
   status: RequestState | null;
   error: Partial<NetworkItem> | null;
}

const statusIcons: Record<RequestState, ReactNode> = {
   [RequestState.OK]: <FontAwesomeIcon icon={faCheck} className={classes.ok} />,
   [RequestState.ERROR]: <FontAwesomeIcon icon={faExclamationTriangle} className={classes.error} />,
   [RequestState.PENDING]: <Spin size={'xxs'} progress={true} />, // TODO: заменить на common 8
};

export const DeployTicketActionStatus: React.FC<Props> = React.memo(({ status, ticketId, patchIds, action, error }) => {
   const statusIcon = useMemo(() => status && statusIcons[status], [status]);

   const { deployTicket } = useDeployTicket(ticketId, true);

   const onOpenModal = useCallback(() => {
      if (error?.error) {
         modalService.open(YpErrorModal, { error: error.error, request: error.request }).subscribe();
      }
   }, [error]);

   const patchList: string[] = useMemo(
      () => (patchIds.size > 0 ? [...patchIds.values()] : deployTicket?.patches?.map(patch => patch.id) ?? []),
      [deployTicket?.patches, patchIds],
   );

   const patchCaption = patchList.join(', ');

   return (
      <div className={classes.row}>
         <div>{statusIcon}</div>
         <div>
            <strong>{ticketId}</strong>:{' '}
            {deployTicket?.sourceType === TDeployTicketSpec_EDeployTicketSourceType.RELEASE_INTEGRATION &&
            patchIds.size > 0 ? (
               patchCaption
            ) : (
               <span title={patchCaption} className={classes.allPatches}>
                  all patches
               </span>
            )}
         </div>
         <div className={classes.errorInfo}>
            {error && (
               <Button onClick={onOpenModal} view={'action'} size={'s'}>
                  Show error
               </Button>
            )}
         </div>
      </div>
   );
});

DeployTicketActionStatus.displayName = 'DeployTicketActionStatus';
