import { faChevronDown, faChevronRight } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { classNames } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useState } from 'react';
import { LegoCheckBox } from '../../../_lego';

import {
   DeployTicketConverter,
   getDeployTicketActionAvailable,
   getDeployTicketActionLoginAvailable,
   isWaitingForApprove,
   MutableTicketStatuses,
   OptionalDeployTicketColumn,
} from '../../../models/ui';
import { useApprovalPermissions, useApprovalPolicy, useDeployTicket } from '../../../redux';
import { useConfig } from '../../../services';
import { noop } from '../../../utils';
import { ItemMode } from '../model';

import { DeployTicketSecondaryInfo } from './components/DeployTicketSecondaryInfo/DeployTicketSecondaryInfo';
import { DeployTicketPrimaryInfo } from './components/DeployTicketPrimaryInfo/DeployTicketPrimaryInfo';
import { DeployTicketActions } from './components/DeployTicketActions/DeployTicketActions';

import classes from './DeployTicketRow.module.css';

type SelectProps = {
   // выделение тикета
   selected?: boolean; // для mode === 'list'
   toggle?(): void;
};

type Props<Mode extends ItemMode> = {
   mode: Mode;
   ticketId: string;
   excludeColumns?: Set<OptionalDeployTicketColumn>;

   // выделение патчей
   onSelectPatches?: (patches: Set<string>) => void;
   selectedPatches?: Set<string>;
} & (Mode extends 'list' ? SelectProps : { [K in keyof SelectProps]?: never });

type ReactProps = {
   <T extends ItemMode>(...args: Parameters<React.FC<Props<T>>>): ReturnType<React.FC<Props<T>>>;
} & {
   [K in keyof React.FC<any>]: React.FC<any>[K];
};

export const DeployTicketRow: ReactProps = React.memo(
   ({ mode, selected, toggle, ticketId, excludeColumns, selectedPatches, onSelectPatches }) => {
      const inList = mode === 'list';

      const { deployTicket } = useDeployTicket(ticketId);

      const toogleSelection = useCallback(() => {
         if (selected) {
            onSelectPatches?.(new Set());
         }
         toggle?.();
      }, [onSelectPatches, selected, toggle]);

      const stageId = deployTicket?.stageId ?? '';
      // region hooks

      // расхлоп
      const [secondaryVisible, setSecondaryVisible] = useState(!inList);
      const toogleSecondary = inList ? () => setSecondaryVisible(e => !e) : noop;
      // endregion

      const { approvalPolicySpec } = useApprovalPolicy(stageId);
      const approvalPermissions = useApprovalPermissions({
         stageIds: [stageId],
         logins: deployTicket ? [...DeployTicketConverter.getApproveLogins([deployTicket])] : [],
      });

      const { login } = useConfig()!.user;
      // endregion

      // region render
      if (!deployTicket) {
         return null;
      }

      if (!stageId) {
         return null;
      }
      const actionsLoginAvailable = getDeployTicketActionLoginAvailable({
         ticket: deployTicket,
         login,
         approvalPolicySpec,
         approvalPermissions,
      });

      const actionsAvailable = getDeployTicketActionAvailable({
         ticket: deployTicket,
         approvalPolicySpec,
         approvalPermissions,
      });

      const waitingForApprove = isWaitingForApprove(deployTicket, actionsAvailable);

      const firstRow = (
         <div
            className={classNames(classes.row, inList ? classes.rowList : classes.rowSingle)}
            onClick={toogleSecondary}
            role={'button'}
            tabIndex={0}
            data-test={'DeployTicket:Row'}
            data-creation-date={deployTicket.creationDate}
         >
            {inList && (
               <>
                  <div role={'button'} tabIndex={0} onClick={e => e.stopPropagation()}>
                     {MutableTicketStatuses.has(deployTicket.status) && (
                        <LegoCheckBox
                           theme={'normal'}
                           size={'s'}
                           view={'default'}
                           tone={'grey'}
                           checked={selected}
                           onChange={toogleSelection}
                        />
                     )}
                  </div>
                  <div>
                     <FontAwesomeIcon icon={secondaryVisible ? faChevronDown : faChevronRight} />
                  </div>
               </>
            )}
            <DeployTicketPrimaryInfo
               ticketId={ticketId}
               mode={mode}
               excludeColumns={excludeColumns ?? new Set()}
               waitingForApprove={waitingForApprove}
            />

            <div className={classes.buttons}>
               <DeployTicketActions
                  ticketId={ticketId}
                  selectedPatchIds={selectedPatches}
                  availableActions={actionsLoginAvailable}
               />
            </div>
         </div>
      );

      return (
         <div
            className={classNames(
               classes.item,
               inList ? classes.itemList : classes.itemSingle,
               inList && secondaryVisible ? classes.opened : undefined,
            )}
         >
            {firstRow}
            {secondaryVisible && (
               <div className={classNames(inList ? classes.secondaryInfo : undefined)}>
                  <DeployTicketSecondaryInfo
                     mode={mode}
                     ticketId={ticketId}
                     onSelectPatches={onSelectPatches}
                     selectedPatches={selectedPatches}
                     selectedTicket={selected}
                     toggleTicket={toggle}
                  />
               </div>
            )}
         </div>
      );
      // endregion
   },
);

DeployTicketRow.displayName = 'DeployTicketRow';
