import { Button } from '@yandex-cloud/uikit';
import React, { useMemo } from 'react';
import { MutableTicketStatuses, TicketAction } from '../../../../../models/ui';

import { useDeployTicket } from '../../../../../redux';
import { useDeployTicketActions } from '../../../hooks';

import classes from './DeployTicketActions.module.css';

interface Props {
   ticketId: string;
   availableActions: Set<TicketAction>;
   selectedPatchIds?: Set<string>;
}

const primaryActions = new Set([TicketAction.Commit, TicketAction.Approve]);

export const DeployTicketActions: React.FC<Props> = React.memo(
   ({ availableActions, ticketId, selectedPatchIds = new Set<string>() }) => {
      const { deployTicket } = useDeployTicket(ticketId);
      const actions = useDeployTicketActions(ticketId);

      const handles: Record<TicketAction, () => void> = useMemo(
         () =>
            Object.keys(actions).reduce((obj, actionName) => {
               const name = actionName as TicketAction;
               obj[name] = () => actions[name](Array.from(selectedPatchIds));
               return obj;
            }, {} as Record<TicketAction, () => void>),
         [actions, selectedPatchIds],
      );

      if (!deployTicket) {
         return null;
      }

      if (!MutableTicketStatuses.has(deployTicket.status)) {
         return null;
      }

      const patchCount = deployTicket.patches.length;
      const selectedPatches =
         patchCount === 1
            ? deployTicket.patches
            : deployTicket.patches.filter(p => (patchCount === 1 ? true : selectedPatchIds.has(p.id)));

      const canDoAction =
         selectedPatches.length > 0
            ? selectedPatches.every(p => MutableTicketStatuses.has(p.status))
            : MutableTicketStatuses.has(deployTicket.status);

      // порядок важен
      const actionNames = [
         TicketAction.Skip,
         TicketAction.Disapprove,
         TicketAction.Approve,
         TicketAction.Commit,
      ].filter(action => availableActions.has(action));

      return (
         <div
            onClick={e => e.stopPropagation()}
            role={'button'}
            tabIndex={0}
            className={classes.actionsPopup}
            data-e2e={'DeployTicket:Actions'}
         >
            <div className={classes.actions}>
               {actionNames.map(action => {
                  const caption = `${action[0].toUpperCase()}${action.slice(1)}`;
                  return (
                     <div>
                        <Button
                           view={primaryActions.has(action) ? 'action' : 'normal'}
                           size={'s'}
                           disabled={!canDoAction}
                           onClick={handles[action]}
                           width={'max'}
                           extraProps={{ 'data-e2e': `DeployTicket:${caption}` }}
                        >
                           {caption}
                        </Button>
                     </div>
                  );
               })}
            </div>
         </div>
      );
   },
);

DeployTicketActions.displayName = 'DeployTicketActions';
