import { faInfoCircle } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Popover } from '@yandex-cloud/uikit';
import { EMPTY_VALUE } from '@yandex-infracloud-ui/libs';
import React from 'react';

import {
   DeployTicketConverter,
   getApprovalsCount,
   isApprovalRequired,
   isMandatory,
   MutableTicketStatuses,
} from '../../../../../models/ui';
import { useApprovalPermissions, useApprovalPolicy, useDeployTicket } from '../../../../../redux';
import { ApprovalList } from '../../../../common';

import classes from './DeployTicketApprovesInfo.module.css';

interface Props {
   ticketId: string;
}

export const DeployTicketApprovesInfo: React.FC<Props> = React.memo(({ ticketId }) => {
   const { deployTicket } = useDeployTicket(ticketId);
   const stageId = deployTicket?.stageId ?? '';

   const { approvalPolicySpec } = useApprovalPolicy(stageId);
   const approvalPermissions = useApprovalPermissions({
      stageIds: [stageId],
      logins: deployTicket ? [...DeployTicketConverter.getApproveLogins([deployTicket])] : [],
   });

   if (!deployTicket) {
      return null;
   }

   const approvalRequired = isApprovalRequired(deployTicket, approvalPolicySpec);
   const mandatorySpec = isMandatory(approvalPolicySpec);
   const approvalsCount = getApprovalsCount({ ticket: deployTicket, approvalPermissions, approvalPolicySpec });

   const { data: approvalData, approve } = deployTicket.approval;
   const existApproves = approvalData.size > 0;

   if (!(approvalRequired || existApproves)) {
      return <div>{EMPTY_VALUE}</div>;
   }
   return (
      <div>
         {approvalRequired && MutableTicketStatuses.has(deployTicket.status) && (
            <span title={`Approves left: ${approvalsCount.left}, approvals count: ${approvalsCount.needed}`}>
               {approvalsCount.sum} / {approvalsCount.needed} regular
               {mandatorySpec && (
                  <>
                     <br />
                     <span>{approvalsCount.mandatorySum} / 1 mandatory</span>
                  </>
               )}
               <br />
            </span>
         )}
         {existApproves && !MutableTicketStatuses.has(deployTicket.status) && (
            <span>{approvalsCount.sum} Approves </span>
         )}
         {existApproves && (
            <Popover
               hasArrow={true}
               content={
                  <>
                     {approvalRequired && MutableTicketStatuses.has(deployTicket.status) && (
                        <div className={classes.approvesInfo}>
                           <span>
                              Approves left: {approvalsCount.left}, approvals count: {approvalsCount.needed}
                           </span>
                        </div>
                     )}
                     <ApprovalList
                        approvals={Array.from(approvalData.entries()).map(([approvalLogin, data]) => {
                           const { comment, date } = data;
                           return {
                              login: approvalLogin,
                              status: approve.has(approvalLogin) ? 'approved' : 'disapproved',
                              comment: comment ?? '',
                              timestamp: date ? new Date(date) : null,
                           };
                        })}
                     />
                  </>
               }
            >
               <FontAwesomeIcon icon={faInfoCircle} />
            </Popover>
         )}
      </div>
   );
});

DeployTicketApprovesInfo.displayName = 'DeployTicketApprovesInfo';
