import React, { ReactNode, useMemo } from 'react';

import { classNames, EMPTY_VALUE } from '@yandex-infracloud-ui/libs';
import { Link } from 'react-router-dom';
import { Label } from '@yandex-cloud/uikit';

import { TDeployTicketSpec_EDeployTicketSourceType } from '../../../../../proto-typings';
import { useDeployTicket, useRelease } from '../../../../../redux';
import { ExecutionDate } from '../ExecutionDate/ExecutionDate';
import { urlBuilder } from '../../../../../models';
import { Column, columnsList, ItemMode } from '../../../model';
import { DeployTicketApprovesInfo } from '../DeployTicketApprovesInfo/DeployTicketApprovesInfo';
import { DeployTicketStatus } from '../../../DeployTicketStatus/DeployTicketStatus';
import { TicketStatus } from '../../../../../models/ui';
import { CopyWrapper } from '../../../../lib';

import classes from './DeployTicketPrimaryInfo.module.css';

interface Props {
   ticketId: string;
   mode: ItemMode;
   excludeColumns: Set<Column>;

   // динамическое вычисление необходимости апрува на верхнем уровне, после внедрения статуса на бекенде необходимость исчезнет
   waitingForApprove?: boolean;
}

/**
 * Информация о тикете, которая показывается в расхлопе
 */
export const DeployTicketPrimaryInfo: React.FC<Props> = React.memo(
   ({ ticketId, mode, excludeColumns, waitingForApprove = false }) => {
      const { deployTicket } = useDeployTicket(ticketId);

      const inList = mode === 'list';

      const isDraftTicket = deployTicket?.sourceType === TDeployTicketSpec_EDeployTicketSourceType.STAGE_DRAFT;

      const columns: readonly Column[] = useMemo(
         () =>
            excludeColumns && excludeColumns.size > 0 ? columnsList.filter(e => !excludeColumns.has(e)) : columnsList,
         [excludeColumns],
      );

      const { release } = useRelease(deployTicket?.releaseId);

      if (!deployTicket) {
         return null;
      }
      const { releaseId, releaseRuleId, stageId } = deployTicket;

      if (!stageId) {
         return null;
      }

      const idCapture = isDraftTicket ? ticketId.split('-')[0] ?? ticketId : ticketId;
      const content: Record<Column, ReactNode> = {
         id: (
            <div>
               <CopyWrapper text={ticketId} className={classes.id} showByHover={true}>
                  {inList ? (
                     <Link to={urlBuilder.ticketHighlightedPage(deployTicket)}>
                        <div title={ticketId} data-test={'DeployTicket:Link'} className={classes.longName}>
                           {idCapture}
                        </div>
                     </Link>
                  ) : (
                     <span title={ticketId}>{idCapture}</span>
                  )}
               </CopyWrapper>
               {deployTicket?.title && (
                  <div className={classes.longName} title={deployTicket.title}>
                     {deployTicket.title}
                  </div>
               )}
               {isDraftTicket && (
                  <div>
                     <Label>Draft</Label>
                  </div>
               )}
            </div>
         ),
         releaseRule: isDraftTicket ? (
            EMPTY_VALUE
         ) : (
            <Link to={urlBuilder.releaseRulePage(stageId, releaseRuleId)}>{releaseRuleId}</Link>
         ),
         release: isDraftTicket ? (
            EMPTY_VALUE
         ) : (
            <div>
               <Link to={urlBuilder.releasePage(releaseId)}>
                  <div className={classes.longName} title={releaseId}>
                     {releaseId}
                  </div>
               </Link>
               {release?.title && (
                  <div className={classes.longName} title={release.title}>
                     {release.title}
                  </div>
               )}
            </div>
         ),
         stage: <Link to={urlBuilder.stage(stageId)}>{stageId}</Link>,
         status: (
            <div data-e2e={'Ticket:Status'}>
               <DeployTicketStatus status={waitingForApprove ? TicketStatus.WaitingForApprove : deployTicket.status} />
            </div>
         ),
         approves: <DeployTicketApprovesInfo ticketId={ticketId} />,
         date: (
            <ExecutionDate
               start={deployTicket.executionStart ? new Date(deployTicket.executionStart) : null}
               end={deployTicket.executionEnd ? new Date(deployTicket.executionEnd) : null}
               create={deployTicket.creationDate ? new Date(deployTicket.creationDate) : null}
            />
         ),
      };

      const gridColumns = `repeat(${columns.length}, 1fr)`;

      return (
         <div className={classNames(classes.row)} style={{ gridTemplateColumns: gridColumns }}>
            {columns.map(id => (
               <div key={id} style={{ minWidth: `${100 / columns.length}%` }}>
                  {content[id]}
               </div>
            ))}
         </div>
      );
   },
);

DeployTicketPrimaryInfo.displayName = 'DeployTicketPrimaryInfo';
