import { Checkbox } from '@yandex-cloud/uikit';

import { StaffCard } from '@yandex-data-ui/common';
import { classNames, EMPTY_VALUE, formatDate, UserName, UserOrGroupType } from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback } from 'react';
import { Link } from 'react-router-dom';

import { urlBuilder } from '../../../../../models';
import { MutableTicketStatuses } from '../../../../../models/ui';
import { TDeployTicketSpec_EDeployTicketSourceType } from '../../../../../proto-typings';
import { useDeployTicket, useRelease } from '../../../../../redux';
import { DefinitionList, DefinitionListItem } from '../../../../lib';
import { DeployTicketStatus } from '../../../DeployTicketStatus/DeployTicketStatus';
import { ItemMode } from '../../../model';

import classes from './DeployTicketSecondaryInfo.module.css';

interface Props {
   mode?: ItemMode;
   ticketId: string;
   onSelectPatches?: (patches: Set<string>) => void;
   selectedPatches?: Set<string>;

   selectedTicket?: boolean;

   toggleTicket?(): void;
}

const columnsList = ['creation', 'author', 'patches', 'status', 'deployUnit'] as const;
type columnName = typeof columnsList[number];

const excludeDraftColumns: Set<columnName> = new Set(['patches', 'status', 'deployUnit']);

/**
 * Информация о тикете, которая показывается в расхлопе
 */
export const DeployTicketSecondaryInfo: React.FC<Props> = React.memo(
   ({ ticketId, toggleTicket, selectedTicket, onSelectPatches, selectedPatches }) => {
      const { deployTicket } = useDeployTicket(ticketId);

      const patchSize = deployTicket?.patches?.length ?? 0;
      const onSelectPatch = useCallback(
         (id: string) => {
            if (onSelectPatches) {
               const newSelected = new Set(selectedPatches ?? []);
               const checked = Boolean(selectedTicket) || Boolean(selectedPatches?.has(id));
               if (checked) {
                  newSelected.delete(id);
               } else {
                  newSelected.add(id);
               }
               onSelectPatches(newSelected);
               if (toggleTicket) {
                  const all = newSelected.size === patchSize;
                  if ((all && !selectedTicket) || (!all && selectedTicket)) {
                     toggleTicket();
                  }
               }
            }
         },
         [onSelectPatches, toggleTicket, selectedTicket, patchSize, selectedPatches],
      );

      const { release } = useRelease(deployTicket?.releaseId);
      if (!deployTicket) {
         return null;
      }

      const { stageId, patches, creationDate, title, description, sourceType } = deployTicket;

      const isDraftTicket = sourceType === TDeployTicketSpec_EDeployTicketSourceType.STAGE_DRAFT;

      const columns: readonly columnName[] = isDraftTicket
         ? columnsList.filter(e => !excludeDraftColumns.has(e))
         : columnsList;

      const titles: Record<columnName, ReactNode> = {
         creation: 'Creation date',
         author: 'Author',
         patches: (
            <span>
               Patches{' '}
               {patches.length > 0 && (
                  <span style={{ fontWeight: 'normal', color: 'var(--yc-color-text-secondary)' }}>
                     {' '}
                     {patches.length}
                  </span>
               )}
            </span>
         ),
         status: 'Status',
         deployUnit: 'Deploy Unit',
      };

      const content: Record<columnName, ReactNode> = {
         creation: (
            <span data-test={'creationDate'} data-value={creationDate}>
               {formatDate(new Date(creationDate))}
            </span>
         ),
         author: isDraftTicket ? (
            EMPTY_VALUE
         ) : release?.author ? (
            <div className={classes.author}>
               <StaffCard login={release.author}>
                  <UserName value={{ id: release.author, type: UserOrGroupType.People }} />
               </StaffCard>
            </div>
         ) : (
            EMPTY_VALUE
         ),
         patches: (
            <>
               {patches.map(patch => {
                  const { id, status } = patch;
                  const checked = Boolean(selectedTicket) || Boolean(selectedPatches?.has(id));
                  return (
                     <div key={id} className={classes.patch}>
                        {MutableTicketStatuses.has(status) && patches.length > 1 && (
                           <Checkbox
                              size={'m'}
                              checked={checked}
                              onChange={() => {
                                 onSelectPatch(id);
                              }}
                           />
                        )}
                        <div>{id}</div>
                     </div>
                  );
               })}
            </>
         ),
         status: (
            <>
               {patches.map(patch => (
                  <DeployTicketStatus status={patch.status} key={patch.id} />
               ))}
            </>
         ),
         deployUnit:
            patches.length > 0 ? (
               <>
                  {patches.map(patch => (
                     <div key={patch.id}>
                        {patch.deployUnitId ? (
                           <Link to={urlBuilder.stageConfig(stageId, patch.deployUnitId)}>{patch.deployUnitId}</Link>
                        ) : (
                           EMPTY_VALUE
                        )}
                     </div>
                  ))}
               </>
            ) : (
               EMPTY_VALUE
            ),
      };

      const gridColumns = `repeat(${columns.length}, 1fr)`;

      const additionalInfo: DefinitionListItem[] = [
         ...(title ? [{ name: 'Title', content: beautifulText(title) }] : []),
         ...(description ? [{ name: 'Description', content: beautifulText(description) }] : []),
      ];

      return (
         <div className={classes.secondaryInfo}>
            <DefinitionList items={additionalInfo} />
            <div className={classes.info}>
               <div className={classNames(classes.row, classes.header)} style={{ gridTemplateColumns: gridColumns }}>
                  {columns.map(id => (
                     <div key={id}>{titles[id]}</div>
                  ))}
               </div>
               <div className={classes.row} style={{ gridTemplateColumns: gridColumns }}>
                  {columns.map(id => (
                     <div key={id}>{content[id]}</div>
                  ))}
               </div>
            </div>
         </div>
      );
   },
);

DeployTicketSecondaryInfo.displayName = 'DeployTicketSecondaryInfo';

function beautifulText(text: string): string {
   return text.replace(/\n{3,}/g, '\n\n').trim();
}
