import React, { ReactNode, useCallback, useEffect, useMemo, useState } from 'react';

import { faSync } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Link } from '@yandex-cloud/uikit';
import { useLocation } from 'react-router';
import {
   EmptyContainer,
   EmptyContainerType,
   fromQuery,
   isEqual,
   Loader,
   TIMES_IN_MS,
   UpdateTimerMode,
   useUpdateTimer,
} from '@yandex-infracloud-ui/libs';
import { Spin } from 'lego-on-react';
import { useDispatch, useSelector } from 'react-redux';
import { map } from 'rxjs/operators';

import { LegoButton } from '../../../_lego';

import {
   ApprovalPolicyConverter,
   ApprovalPolicySpec,
   DeployTicket,
   DeployTicketAggregation,
   DeployTicketConverter,
   DeployTicketFilters,
   emptyDeployTicketFilters,
   filterDeployTickets,
   filterDeployTicketsByApprovalStatuses,
   getDeployTicketActionAvailable,
   initialDeployTicketFilters,
   MutableTicketStatuses,
   OnlyTicketStatuses,
   prepareFilterQuery,
   TicketStatus,
} from '../../../models/ui';
import { TApprovalPolicySpec } from '../../../proto-typings';
import {
   loadReleasesData,
   RootState,
   selectApprovalPolicies,
   selectContinuationTokens,
   SelectDeployTicketOptions,
   selectDeployTickets,
   useApprovalPermissions,
   useNetworkErrors,
   useNetworkRequests,
} from '../../../redux';
import { RequestState } from '../../../redux/slices/network/model';
import { ypApi } from '../../../services';
import { GetDeployTicketsReqOptions } from '../../../services/api/services/YpApi';
import { pluralNumber } from '../../../utils';
import { Badge, BadgeType } from '../../lib';
import { YpErrorTooltip } from '../../network';
import { DeployTicketFiltersView, parseFiltersFromUrl } from '../DeployTicketFiltersView/DeployTicketFiltersView';
import { DeployTicketList } from '../DeployTicketList/DeployTicketList';
import { DeployTicketsNotFound } from '../DeployTicketsNotFound/DeployTicketsNotFound';
import { useReleaseLoadData } from '../hooks';
import { Column, DeployTicketPageType } from '../model';

import classes from './DeployTicketsBlock.module.css';

const namespace = 'DeployTicketsBlock';
const requestKeys = {
   filter: `${namespace}/filter`,
   load: `${namespace}/load`,
   update: `${namespace}/update`,
};
const LOAD_LIMIT = 25;

interface Props {
   pageType: DeployTicketPageType;
   headerPanel?: ReactNode;
   selectOptions?: SelectDeployTicketOptions;
   requestOptions?: GetDeployTicketsReqOptions;
   defaultFilters?: Partial<DeployTicketFilters>;
   excludeColumns?: Set<Column>;
   emptyBlock?: ReactNode;
}

const initialFiltersWithHistoryStatuses: DeployTicketFilters = {
   releaseRuleIdsAndDrafts: new Set(),
   ticketStatuses: new Set(OnlyTicketStatuses),
   ticketId: '',
   releaseId: '',
   stageId: '',
};

export const DeployTicketsBlock: React.FC<Props> = React.memo(
   ({ pageType, headerPanel, selectOptions, requestOptions = {}, defaultFilters, excludeColumns, emptyBlock }) => {
      const requests = useNetworkRequests(Object.values(requestKeys));
      const errors = useNetworkErrors(Object.values(requestKeys));
      const { search } = useLocation();

      const { limitInTest } = fromQuery(search);
      const limit = limitInTest ? Number(limitInTest) : LOAD_LIMIT;

      const isGlobalLoading = requests[requestKeys.filter]?.state === RequestState.PENDING;
      const isUpdateLoading = requests[requestKeys.update]?.state === RequestState.PENDING;
      const isTailLoading = requests[requestKeys.load]?.state === RequestState.PENDING;

      const deployTicketsSelector = useCallback((s: RootState) => selectDeployTickets(s, selectOptions ?? {}), [
         selectOptions,
      ]);
      const deployTickets = useSelector(deployTicketsSelector);

      const initialFiltersFromUrl = useMemo(() => parseFiltersFromUrl(search), [search]);
      const initialDefaultFilters = useMemo(
         () => ({
            ...(pageType === DeployTicketPageType.release
               ? initialFiltersWithHistoryStatuses
               : initialDeployTicketFilters),
         }),
         [pageType],
      );

      const [filters, setFilters] = useState<DeployTicketFilters>(() => ({
         ...initialDefaultFilters,
         ...initialFiltersFromUrl,
      }));
      const isUsingFilters = useMemo(() => !isEqual(filters, emptyDeployTicketFilters), [filters]);

      const loadedStageIds = useMemo(() => [...DeployTicketAggregation.getStageIds(deployTickets)], [deployTickets]);

      const approvalPermissions = useApprovalPermissions({
         stageIds: loadedStageIds,
         logins: [...DeployTicketConverter.getApproveLogins(deployTickets)],
         skipLoad: true,
      });

      const approvalSelector = useCallback((state: RootState) => selectApprovalPolicies(state, loadedStageIds), [
         loadedStageIds,
      ]);
      const approvalPolicies = useSelector(approvalSelector);

      const approvalPoliciesSpec = useMemo(
         () =>
            Object.keys(approvalPolicies).reduce((specs, id) => {
               const rawPolicy = approvalPolicies[id];
               if (!rawPolicy) {
                  return specs;
               }
               const spec = ApprovalPolicyConverter.getSpec(rawPolicy.spec as TApprovalPolicySpec);
               if (spec) {
                  specs[id] = spec;
               }
               return specs;
            }, {} as Record<string, ApprovalPolicySpec>),
         [approvalPolicies],
      );

      const getActionAvailable = useCallback(
         (ticket: DeployTicket) => {
            const approvalPolicySpec = approvalPoliciesSpec[ticket.stageId] ?? null;
            return getDeployTicketActionAvailable({ ticket, approvalPermissions, approvalPolicySpec });
         },
         [approvalPermissions, approvalPoliciesSpec],
      );

      const approveFilter = filters.ticketStatuses.has(TicketStatus.WaitingForApprove);
      const commitFilter = filters.ticketStatuses.has(TicketStatus.WaitingForCommit);

      const filteredDeployTickets = useMemo(
         () =>
            deployTickets
               .filter(filterDeployTickets(filters))
               .filter(filterDeployTicketsByApprovalStatuses(approveFilter, commitFilter, getActionAvailable)),
         [approveFilter, commitFilter, deployTickets, filters, getActionAvailable],
      );

      const dispatch = useDispatch();

      const { allReleaseIds, allStageIds, allStageLogins } = useReleaseLoadData({
         tickets: filteredDeployTickets,
         filters,
      });

      const requestDeployTicketData: GetDeployTicketsReqOptions = useMemo(
         () => ({
            ...requestOptions,
            filterQueryParams: prepareFilterQuery(filters),
         }),
         [filters, requestOptions],
      );

      const loadAllIds = useMemo(
         () =>
            ypApi
               .getDeployTickets({
                  ...requestOptions,
                  filterQueryParams: prepareFilterQuery(filters),
                  limit: 10000, // считаем, что это сильно выше любого лимита
                  paths: e => [e.meta.id],
                  withOrder: false,
               })
               .pipe(map(tickets => new Set(tickets.values.map(ticket => ticket.meta!.id!)))),
         [filters, requestOptions],
      );

      const handleLoadingTickets = useCallback(
         ({ key, reset = false, continuationToken }) => {
            dispatch(
               loadReleasesData.withMeta({ reset }).withRequestKey(key)(
                  {
                     ...requestDeployTicketData,
                     limit,
                     continuationToken,
                  },
                  { stageIds: allStageIds, releaseIds: allReleaseIds, stageLogins: allStageLogins },
               ),
            );
         },
         [allReleaseIds, allStageIds, allStageLogins, dispatch, limit, requestDeployTicketData],
      );

      // для новых данных по новым фильтрам
      useEffect(() => {
         handleLoadingTickets({ key: requestKeys.filter, reset: true });
         // eslint-disable-next-line react-hooks/exhaustive-deps
      }, [filters]);

      // для подгрузки тикетов
      const continuationTokens = useSelector(selectContinuationTokens);

      const loadMore = useCallback(
         () => handleLoadingTickets({ key: requestKeys.load, continuationToken: continuationTokens.tickets }),
         [continuationTokens.tickets, handleLoadingTickets],
      );

      // обновление данных
      const handleRefresh = useCallback(() => handleLoadingTickets({ key: requestKeys.update }), [
         handleLoadingTickets,
      ]);

      useUpdateTimer({
         callback: handleRefresh,
         slow: TIMES_IN_MS.Minute,
         fast: TIMES_IN_MS.Minute,
         mode: UpdateTimerMode.Slow,
      });

      const globalError = isGlobalLoading ? null : errors[requestKeys.filter];
      const tailError = isTailLoading ? null : errors[requestKeys.load];

      const updatedTicketCount = useMemo(
         () => filteredDeployTickets.filter(ticket => MutableTicketStatuses.has(ticket.status)).length,
         [filteredDeployTickets],
      );

      const refreshTitle = 'Refresh tickets';

      return (
         <div>
            <div className={classes.toolbar}>
               <DeployTicketFiltersView
                  filters={filters}
                  setFilters={setFilters}
                  excludeColumns={excludeColumns}
                  defaultFilters={defaultFilters}
               />

               <div className={classes.separator} />

               <div className={classes.buttonRow}>
                  <div data-test={'force-refresh'}>
                     <LegoButton
                        theme={'normal'}
                        size={'s'}
                        onClick={handleRefresh}
                        title={refreshTitle}
                        disabled={isGlobalLoading || isUpdateLoading}
                     >
                        <FontAwesomeIcon icon={faSync} spin={isGlobalLoading || isUpdateLoading} />
                     </LegoButton>
                  </div>
                  {headerPanel}
               </div>
            </div>

            <div className={classes.badgeContainer}>
               {updatedTicketCount > 0 && (
                  <Badge type={BadgeType.Warning} className={classes.badge}>
                     {pluralNumber(updatedTicketCount, 'ticket waits an action', 'tickets wait an action')}
                  </Badge>
               )}
            </div>

            {isGlobalLoading ? (
               <div className={classes.globalLoader}>
                  <Spin size={'l'} progress={true} />
               </div>
            ) : globalError?.error ? (
               <EmptyContainer
                  type={EmptyContainerType.Error}
                  title={'Network error'}
                  description={<YpErrorTooltip error={globalError.error} request={globalError.request} />}
               />
            ) : (
               <>
                  {filteredDeployTickets.length > 0 ? (
                     <div>
                        <DeployTicketList pageType={pageType} tickets={filteredDeployTickets} loadAllIds={loadAllIds} />

                        {continuationTokens?.tickets && (
                           <div className={classes.loadMore} data-test={'load-more-tickets'}>
                              {isTailLoading ? (
                                 <Loader inline={true} />
                              ) : (
                                 <Link view={'normal'} onClick={loadMore}>
                                    Load More Tickets
                                 </Link>
                              )}
                              {tailError?.error && (
                                 <YpErrorTooltip error={tailError.error} request={tailError.request} />
                              )}
                           </div>
                        )}
                     </div>
                  ) : isUsingFilters ? (
                     <DeployTicketsNotFound />
                  ) : (
                     emptyBlock
                  )}
               </>
            )}
         </div>
      );
   },
);

DeployTicketsBlock.displayName = 'DeployTicketsBlock';
