import { DISMISS_REASON, modalService, toasts, useDismounted } from '@yandex-infracloud-ui/libs';
import { useCallback, useMemo } from 'react';
import { takeUntil } from 'rxjs/operators';
import { useDispatch } from 'react-redux';
import { DeployTicketActionModal, TicketActionModalLabels } from '../DeployTicketActionModal/DeployTicketActionModal';
import { useRelease } from '../../../redux/hooks/useRelease';
import { fetchDeployTicket, useDeployTicket } from '../../../redux';
import { getPreferredTicketDescription } from '../model';
import { TicketAction } from '../../../models/ui';

export type DeployTicketAction = (patchIds: string[]) => void;
export type DeployTicketActions = Record<TicketAction, DeployTicketAction>;

const modalLabels: Record<TicketAction, TicketActionModalLabels> = {
   [TicketAction.Commit]: {
      title: 'Commit ticket',
      actionInProcess: 'Ticket is committing',
      submitButton: 'Commit',
   },
   [TicketAction.Skip]: {
      title: 'Skip ticket',
      actionInProcess: 'Ticket is skipping',
      submitButton: 'Skip',
   },
   [TicketAction.Approve]: {
      title: 'Approve ticket',
      actionInProcess: 'Ticket is approving',
      submitButton: 'Approve',
   },
   [TicketAction.Disapprove]: {
      title: 'Disapprove ticket',
      actionInProcess: 'Ticket is disapproving',
      submitButton: 'Disapprove',
   },
};

export function useDeployTicketActions(ticketId: string): DeployTicketActions {
   const dismounted = useDismounted();
   const dispatch = useDispatch();

   const { deployTicket } = useDeployTicket(ticketId);

   const releaseId = deployTicket?.releaseId ?? '';
   const { release } = useRelease(releaseId);

   const loadTicket = useCallback(() => dispatch(fetchDeployTicket(ticketId ?? '')), [dispatch, ticketId]);

   const getAction: (action: TicketAction) => DeployTicketAction = useCallback(
      action => patchIds => {
         if (!ticketId) {
            return;
         }
         modalService
            .open(DeployTicketActionModal, {
               action,
               initialMessage: getPreferredTicketDescription(action, deployTicket, release),
               labels: modalLabels[action],
               patchIds,
               ticketId,
            })
            .pipe(takeUntil(dismounted))
            .subscribe(loadTicket, reason => {
               if (reason !== DISMISS_REASON) {
                  toasts.apiError('Skipping the ticket', reason);
               }
            });
      },
      [deployTicket, dismounted, loadTicket, release, ticketId],
   );

   return useMemo(
      () => ({
         [TicketAction.Commit]: getAction(TicketAction.Commit),
         [TicketAction.Skip]: getAction(TicketAction.Skip),
         [TicketAction.Approve]: getAction(TicketAction.Approve),
         [TicketAction.Disapprove]: getAction(TicketAction.Disapprove),
      }),
      [getAction],
   );
}
