import { useEffect, useMemo, useState } from 'react';
import { DeployTicket, DeployTicketAggregation, DeployTicketFilters } from '../../../models/ui';
import { useConfig } from '../../../services';

interface Params {
   tickets: DeployTicket[];
   filters: DeployTicketFilters;
}

interface idData {
   allStageIds: Set<string>;
   allReleaseIds: Set<string>;
   allStageLogins: Set<string>;
}

/**
 * Считает id стейджей и релизов для заданного списка тикетов
 */
export function useReleaseLoadData({ tickets, filters }: Params): idData {
   const [allStageIds, setAllStageIds] = useState<Set<string>>(new Set());
   const [allReleaseIds, setAllReleaseIds] = useState<Set<string>>(new Set());
   const [allStageLogins, setAllStageLogins] = useState<Set<string>>(new Set());

   const stageIds = useMemo(() => DeployTicketAggregation.getStageIds(tickets), [tickets]);
   const releaseIds = useMemo(() => DeployTicketAggregation.getReleaseIds(tickets), [tickets]);

   const { login } = useConfig()!.user;
   const stageLogins = useMemo(() => DeployTicketAggregation.getApprovalStageLogins(tickets, [login]), [
      login,
      tickets,
   ]);

   // сброс id при изменении фильтров
   useEffect(() => {
      setAllReleaseIds(new Set());
      setAllStageIds(new Set());
      setAllStageLogins(new Set());
   }, [filters]);

   // перерасчёт id
   useEffect(() => {
      setAllStageIds(allIds => {
         const existIds = new Set(allIds);
         for (const id of stageIds) {
            existIds.add(id);
         }
         return existIds;
      });
   }, [stageIds]);

   useEffect(() => {
      setAllReleaseIds(allIds => {
         const existIds = new Set(allIds);
         for (const id of releaseIds) {
            existIds.add(id);
         }
         return existIds;
      });
   }, [releaseIds]);

   useEffect(() => {
      setAllStageLogins(allIds => {
         const existIds = new Set(allIds);
         for (const id of stageLogins) {
            existIds.add(id);
         }
         return existIds;
      });
   }, [stageLogins]);

   return {
      allStageIds,
      allReleaseIds,
      allStageLogins,
   };
}
