import { DeployTicket, Release, TicketAction } from '../../models/ui';

export const columnsList = ['id', 'stage', 'release', 'releaseRule', 'status', 'approves', 'date'] as const;

export type Column = typeof columnsList[number];

export type ItemMode = 'single' | 'list';

export function getPreferredTicketDescription(
   action: TicketAction,
   ticket: DeployTicket | null,
   release?: Release | null,
): string {
   const description = release?.description || ticket?.description || release?.title || '';

   const initialMessages: Partial<Record<TicketAction, string>> = {
      [TicketAction.Commit]: description,
      [TicketAction.Skip]: description,
   };

   return initialMessages[action] ?? '';
}

export enum DeployTicketPageType {
   stage = 'stage',
   release = 'release',
}

export interface DeployTicketDUGroup {
   /** тикеты, которые патчат текущий деплой юнит */
   ticketIds: Set<string>;
}
export interface DeployTicketStageGroup {
   deployUnitGroups: Record<string, DeployTicketDUGroup>;
   ticketIds: Set<string>;

   /** тикеты для стейджа, для которых деплой юнит неизвестен */
   unknownTicketIds: Set<string>;
}

export interface DeployTicketGroups {
   ticketsByStage: Record<string, DeployTicketStageGroup>;

   /** тикеты, для которых стейдж неизвестен (например, известны только id тикетов для релиза) */
   unknownTicketIds: Set<string>;

   stagesWithMultiTickets: Set<string>;
}

export function getDeployTicketGroups(ticketsMap: Map<string, DeployTicket | null>): DeployTicketGroups {
   const groups: DeployTicketGroups = {
      ticketsByStage: {},
      unknownTicketIds: new Set(),
      stagesWithMultiTickets: new Set(),
   };

   for (const [ticketId, ticket] of ticketsMap) {
      if (ticket === null) {
         groups.unknownTicketIds.add(ticketId);
      } else {
         const { stageId, patches } = ticket;

         if (!(stageId in groups.ticketsByStage)) {
            groups.ticketsByStage[stageId] = {
               deployUnitGroups: {},
               unknownTicketIds: new Set(),
               ticketIds: new Set(),
            };
         }

         const stageGroup = groups.ticketsByStage[stageId];

         stageGroup.ticketIds.add(ticketId);

         for (const patch of patches) {
            const { deployUnitId } = patch;
            if (deployUnitId === null) {
               stageGroup.unknownTicketIds.add(ticketId);
            } else {
               if (!(deployUnitId in stageGroup.deployUnitGroups)) {
                  stageGroup.deployUnitGroups[deployUnitId] = {
                     ticketIds: new Set(),
                  };
               }

               const duGroup = stageGroup.deployUnitGroups[deployUnitId];
               duGroup.ticketIds.add(ticketId);

               if (duGroup.ticketIds.size > 1 && !groups.stagesWithMultiTickets.has(stageId)) {
                  groups.stagesWithMultiTickets.add(stageId);
               }
            }
         }
      }
   }

   return groups;
}
