import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { ExtendedFieldConfig, json } from '@yandex-infracloud-ui/libs';
import { Formik, FormikConfig } from 'formik';
import { Button } from 'lego-on-react';
import React from 'react';
import { array, object, string } from 'yup';

import { CheckboxListField } from './CheckboxListField';

const initialValues = {
   empty: [] as string[],
   regular: ['xxx'],
   withError: [] as string[],
};

type FormValue = typeof initialValues;

function CheckboxListFieldStory(props: ExtendedFieldConfig<FormValue, string[], any>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true,
      } as any,
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         empty: array().of(string()),
         regular: array().of(string()),
         withError: array().min(2).required(),
      }),
   };

   return (
      <Formik {...config}>
         {form => (
            <>
               <CheckboxListField {...props} />

               <Button theme={'action'} size={'s'} type={'submit'} text={'Submit'} />

               <pre>{json(form)}</pre>
            </>
         )}
      </Formik>
   );
}

const controlProps = {
   options: [
      { value: 'xxx', title: 'XXX' },
      { value: 'yyy', title: 'YYY' },
      { value: 'zzz', title: 'ZZZ' },
   ],
};

export const Regular = () => <CheckboxListFieldStory name={'regular'} label={'Regular'} controlProps={controlProps} />;

export const Disabled = () => (
   <CheckboxListFieldStory name={'regular'} label={'Disabled'} controlProps={controlProps} disabled={true} />
);

export const Readonly = () => (
   <CheckboxListFieldStory name={'regular'} label={'Readonly'} controlProps={controlProps} readonly={true} />
);

export const Empty = () => (
   <CheckboxListFieldStory name={'empty'} label={'Readonly & Empty'} controlProps={controlProps} readonly={true} />
);

export const WithError = () => (
   <CheckboxListFieldStory name={'withError'} label={'With error'} controlProps={controlProps} />
);

export const Playground = () => (
   <CheckboxListFieldStory
      name={'regular'}
      label={text('label', 'label')}
      controlProps={controlProps}
      required={boolean('required', false)}
      disabled={boolean('disabled', false)}
      readonly={boolean('readonly', false)}
      help={text('help', 'help text')}
      hint={text('hint', 'short hint')}
      hidden={boolean('hidden', false)}
      placeholder={text('placeholder', 'placeholder')}
   />
);

export default {
   title: 'components/forms/fields/CheckboxField',
} as Meta;
