import { action } from '@storybook/addon-actions';
import { boolean, text } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { ExtendedFieldConfig } from '@yandex-infracloud-ui/libs';
import { Formik, FormikConfig } from 'formik';
import React from 'react';
import { array, object, string } from 'yup';

import { LegoButton } from '../../../../_lego';
import { DevForm } from '../../../lib';

import { KeyValueField } from './KeyValueField';
import { KeyValue } from './models';

const initialValues = {
   default: [
      {
         key: 'key',
         value: 'value',
      },
   ] as KeyValue[],
   withError: null,
};

type FormValue = typeof initialValues;

function FieldStory(props: ExtendedFieldConfig<FormValue>) {
   const config: FormikConfig<FormValue> = {
      initialTouched: {
         withError: true as any,
      },
      initialValues,
      onSubmit: action('onSubmit'),
      validateOnMount: true,
      validationSchema: object({
         withError: array().of(string()).required(),
         xField: array().of(string()).min(10),
      }),
   };

   return (
      <Formik {...config}>
         {() => (
            <div style={{ ['--forms-field-layout' as any]: 'column' }}>
               <KeyValueField placeholder={'Placeholder'} {...props} />

               <LegoButton theme={'action'} type={'submit'} text={'Submit'} />

               <DevForm />
            </div>
         )}
      </Formik>
   );
}

export const Default = () => <FieldStory name={'default'} label={'Default'} />;

export const Disabled = () => <FieldStory name={'default'} label={'Disabled'} disabled={true} />;

export const Readonly = () => <FieldStory name={'default'} label={'Readonly'} readonly={true} />;

export const WithError = () => <FieldStory name={'withError'} label={'With error'} />;

export const Playground = () => (
   <FieldStory
      name={'default'}
      label={text('label', 'label')}
      placeholder={text('placeholder', 'placeholder')}
      required={boolean('required', false)}
      disabled={boolean('disabled', false)}
      readonly={boolean('readonly', false)}
      help={text('help', 'help text')}
      hint={text('hint', 'short hint')}
      hidden={boolean('hidden', false)}
      controlProps={{
         addButton: text('addButton', 'Add record'),
         keyLabel: text('keyLabel', 'Key label'),
         valueLabel: text('valueLabel', 'Value label'),
      }}
   />
);

export default {
   title: 'components/forms/fields/KeyValueField',
} as Meta;
