import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { Table, TableColumnConfig } from '@yandex-cloud/uikit';
import {
   EMPTY_VALUE,
   ExtendedFieldConfig,
   FieldLayout2,
   FormButton,
   InputField2,
   isEmpty,
   RemoveButton,
   useExtendedField,
} from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback, useMemo } from 'react';

import classes from './KeyValueField.module.css';
import { KeyValue, KeyValueControlProps } from './models';

interface TableRow {
   key: ReactNode;
   value: ReactNode;
   buttons: ReactNode;
}

export const KeyValueField = React.memo((props: ExtendedFieldConfig<any, KeyValue[], KeyValueControlProps>) => {
   const { field, readonly = false, disabled = false, onChange } = useExtendedField(props);
   const { name, value } = field;

   const { keyLabel = 'Key', valueLabel = 'Value', addButton = 'Add' } = props.controlProps ?? {};

   const columns: TableColumnConfig<TableRow>[] = useMemo(
      () => [
         { id: 'key', name: keyLabel },
         { id: 'value', name: valueLabel },
         { id: 'buttons', name: '' },
      ],
      [keyLabel, valueLabel],
   );

   const handleRemove = useCallback(
      (i: number) => {
         const newValue: KeyValue[] = [...value];
         newValue.splice(i, 1);
         onChange(newValue);
      },
      [value, onChange],
   );

   const handleAddRow = useCallback(() => {
      const newValue: KeyValue[] = [...value, { key: '', value: '' }];
      onChange(newValue);
   }, [value, onChange]);

   if (readonly) {
      return (
         <FieldLayout2 {...props}>
            {value && !isEmpty(value) ? (
               value.map((row, i) => (
                  // eslint-disable-next-line react/no-array-index-key
                  <FieldLayout2
                     key={i.toString()}
                     name={`${name}.[${i}]`}
                     label={row.key}
                     readonly={true}
                     readonlyDots={true}
                  >
                     <div className={classes.readonly}>{row.value}</div>
                  </FieldLayout2>
               ))
            ) : (
               <>{EMPTY_VALUE}</>
            )}
         </FieldLayout2>
      );
   }

   return (
      <FieldLayout2
         {...props}
         // выводим общую ошибку массива, если массив пустой
         // если не пустой, то выводим ошибки в строках под полями
         hideErrors={value && !isEmpty(value)}
      >
         {value && !isEmpty(value) ? (
            <Table
               columns={columns}
               data={value.map((row, i) => ({
                  'key': (
                     <div className={classes.key}>
                        <InputField2
                           name={`${field.name}[${i}].key`}
                           label={null}
                           placeholder={`Enter ${keyLabel}`}
                           disabled={disabled}
                        />
                     </div>
                  ),
                  'value': (
                     <div className={classes.value}>
                        <InputField2
                           name={`${field.name}[${i}].value`}
                           label={null}
                           placeholder={`Enter ${keyLabel}`}
                           disabled={disabled}
                        />
                     </div>
                  ),
                  buttons: <>{!disabled && !readonly ? <RemoveButton onClick={() => handleRemove(i)} /> : null}</>,
               }))}
               verticalAlign={'top'}
            />
         ) : null}

         {disabled || readonly ? null : (
            <FormButton onClick={handleAddRow} skipLeftSpace={true} icon={faPlus}>
               {addButton}
            </FormButton>
         )}
      </FieldLayout2>
   );
});

KeyValueField.displayName = 'KeyValueField';
