import { YCSelect, YCSelectItemValue, YCSelectProps, YCSelectType } from '@yandex-data-ui/common';
import {
   classNames,
   EMPTY_VALUE,
   ExtendedFieldConfig,
   FieldLayout2,
   useExtendedField,
} from '@yandex-infracloud-ui/libs';
import React, { ReactNode, useCallback } from 'react';

import { useRafDelay } from '../../../../utils';

import classes from './YCSelectField.module.css';

interface YCSelectFieldControlProps<T extends YCSelectType>
   extends Omit<YCSelectProps<T>, 'onChange' | 'disabled' | 'value'> {
   getReadonlyValue?(v: any): ReactNode;
}

type Props<T extends YCSelectType> = ExtendedFieldConfig<any, any, Omit<YCSelectFieldControlProps<T>, 'onUpdate'>>;

/**
 * Довольно низкоуровневая обертка над YCSelect.
 * Все необходимое для работы YCSelect придется передавать в controlProps.
 */
export function YCSelectField<T extends YCSelectType>(props: Props<T>) {
   const { field, onChange, onBlur, readonly, disabled } = useExtendedField(props);

   const { getReadonlyValue, getItems, ...ycSelectProps } = props.controlProps ?? {};

   // Нельзя вызвать onBlur сразу после onChange (изменения еще не применились)
   // Откладываю на следующий тик анимации
   const delayedBlur = useRafDelay(onBlur);

   const renderReadonly = useCallback(
      (v: any) => (!v ? (EMPTY_VALUE as ReactNode) : getReadonlyValue ? getReadonlyValue(v) : String(v)),
      [getReadonlyValue],
   );

   const handleChange = useCallback(
      (rawValue: YCSelectItemValue<T>) => {
         onChange(rawValue);
         delayedBlur();
      },
      [onChange, delayedBlur],
   );

   return (
      <FieldLayout2 {...props}>
         {readonly ? (
            <div className={classes.readonlyValue}>{renderReadonly(field.value)}</div>
         ) : (
            <YCSelect
               {...ycSelectProps}
               className={classNames(classes.control, props['data-test'] ? `test-${props['data-test']}` : undefined)}
               disabled={disabled}
               getItems={getItems}
               onUpdate={handleChange}
               value={field.value}
            />
         )}
      </FieldLayout2>
   );
}

YCSelectField.displayName = 'YCSelectField';
