/* eslint-disable react/sort-comp */
import { classNames, IDismountedProps, IListOption, toggleSetItem, withDismounted } from '@yandex-infracloud-ui/libs';
import { bool, func, instanceOf, string } from 'prop-types';
import * as React from 'react';
import { KeyboardEvent, SyntheticEvent } from 'react';
import { Observable } from 'rxjs';
import { takeUntil } from 'rxjs/operators';

import { LegoTextInput } from '../../../../_lego';
import { TagList } from '../TagList/TagList';

import styles from './TagInput.module.css';

interface Props {
   className?: string;
   disabled?: boolean;
   id?: string;
   name?: string;
   placeholder?: string;
   value: Set<string>;

   onChange(e: SyntheticEvent | null, v: Set<string>): void;

   onBlur?(): void;

   getSuggestions?(term: string): Observable<string[]>;
}

interface IState {
   term: string;
}

class TagInput extends React.PureComponent<Props & IDismountedProps, IState> {
   public state: IState = {
      term: '',
   };

   constructor(props: Props) {
      super(props);

      this.dataProvider = this.dataProvider.bind(this);
      this.handleRemove = this.handleRemove.bind(this);
      this.onEdit = this.onEdit.bind(this);
      this.onKeyDown = this.onKeyDown.bind(this);
      this.tryToAdd = this.tryToAdd.bind(this);
   }

   // noinspection JSUnusedLocalSymbols

   public render() {
      const items: IListOption[] = Array.from(this.props.value).map(v => ({ name: v, value: v }));

      return (
         <div className={classNames(this.props.className)}>
            {!this.props.disabled ? (
               <LegoTextInput
                  id={this.props.id}
                  name={this.props.name}
                  placeholder={this.props.placeholder}
                  text={this.state.term}
                  onChange={this.onEdit}
                  onKeyDown={this.onKeyDown}
                  onBlur={this.tryToAdd}
                  hasClear={true}
                  suggest={Boolean(this.props.getSuggestions)}
                  showListOnFocus={true}
                  suggestUrl={''}
                  dataprovider={this.dataProvider}
                  controlAttrs={{ autoComplete: 'off' }}
               />
            ) : null}

            {items.length > 0 ? (
               <TagList
                  className={classNames({ [styles.editable]: !this.props.disabled })}
                  items={items}
                  onRemove={this.handleRemove}
                  editable={!this.props.disabled}
               />
            ) : null}
         </div>
      );
   }

   private dataProvider(url: string, text: string, cb: (result: string[]) => void): void {
      this.props.getSuggestions!(text)
         .pipe(takeUntil(this.props.dismounted!))
         .subscribe(suggestions => {
            const unused = suggestions.filter(s => !this.props.value.has(s));

            return cb(unused);
         });
   }

   private handleRemove(id: string) {
      this.props.onChange(null, toggleSetItem(this.props.value, id));
   }

   private onEdit(term: string): void {
      this.setState({ term });
   }

   private onKeyDown(e: KeyboardEvent) {
      switch (e.key) {
         case 'Enter':
         case ',':
         case ' ':
            e.preventDefault();
            this.tryToAdd(e);
            break;

         default:
            // Пропускаем событие дальше
            break;
      }
   }

   private tryToAdd(e: SyntheticEvent) {
      if (this.props.onBlur) {
         this.props.onBlur();
      }
      // Предотвращаю удаления тега при его повторном вводе
      if (this.props.value.has(this.state.term)) {
         return;
      }

      // Нечего добавлять
      if (!this.state.term.trim()) {
         return;
      }

      const value = new Set(this.props.value);
      value.add(this.state.term);

      this.props.onChange(e, value);
      this.setState({ term: '' });
   }
}

// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore
TagInput.propTypes = {
   disabled: bool,
   getSuggestions: func,
   id: string,
   name: string,
   onChange: func.isRequired,
   placeholder: string,
   value: instanceOf(Set).isRequired as any, // FIXME remove as any
};

// eslint-disable-next-line @typescript-eslint/ban-ts-comment
// @ts-ignore
TagInput.defaultProps = {
   disabled: false,
   id: '',
   placeholder: 'Type new tag',
};

export const TagInputEnhanced = withDismounted(TagInput);
