import { isEqual } from '@yandex-infracloud-ui/libs';
import { connect as connectFormik, FormikProps, FormikState } from 'formik';
import React, { ReactElement } from 'react';
import { connect as connectRedux, DispatchProp } from 'react-redux';

import type { RootState } from '../../../../redux';
import { pickFields } from '../../../../utils';

import { formStatesSlice, FormStoreRecord, ReducedFormikState, selectCurrentFormRecord } from './formStatesSlice';

interface RenderProps {
   lastChange: Date | null;

   revert(): void;
}

interface Props {
   children: (renderProps: RenderProps) => ReactElement | null;
   formId: string;

   onChange?(formId: string, newValues: any, oldValues: any): void;

   /**
    * Опциональный обработчик "события" отката форма
    */
   onRevert?(formId: string): void;
}

function extractFormikState<T>(form: FormikProps<T>): ReducedFormikState {
   return pickFields(form, ['values', 'errors', 'touched']);
}

interface StateProps {
   formRecord: FormStoreRecord | undefined;
}

type PropsWithHOCs = Props & DispatchProp & StateProps & { formik: FormikProps<any> };

interface State {
   lastChange: Date | null;
}

/**
 * Сделал на классах, т.к. тут сложно на хуках отработать корректно жизненный цикл
 */
class FormLocalStateImpl extends React.Component<PropsWithHOCs, State> {
   private static isEqualState(f1: FormikProps<any>, f2: FormikProps<any>): boolean {
      return isEqual(extractFormikState(f1), extractFormikState(f2));
   }

   constructor(props: PropsWithHOCs) {
      super(props);

      this.state = {
         lastChange: null,
      };

      this.revert = this.revert.bind(this);
   }

   public componentDidMount() {
      this.loadFromStore();
   }

   public componentDidUpdate(prevProps: PropsWithHOCs) {
      // Форма целиком изменилась (была переключена на другую)
      if (this.props.formId !== prevProps.formId) {
         this.setState({ lastChange: null });
         this.loadFromStore();

         return;
      }

      // состояние формы было подгружено извне
      if (prevProps.formRecord === undefined && this.props.formRecord?.updateType === 'targetValue') {
         this.loadFromStore();
         this.props.onChange?.(this.props.formId, this.props.formik.values, prevProps.formik.values);
         return;
      }

      // Состояние формы изменилось
      if (!FormLocalStateImpl.isEqualState(prevProps.formik, this.props.formik)) {
         this.updateForm(this.props.formId, this.props.formik, prevProps.formik);
      }
   }

   public revert() {
      this.props.formik.resetForm();

      this.removeFromStore(this.props.formId);

      this.props.onRevert?.(this.props.formId);
   }

   public render() {
      return this.props.children({
         lastChange: this.state.lastChange,
         revert: this.revert,
      });
   }

   private dispatch(action: any) {
      // Откладываю на чуть-чуть выполнение кода, чтобы не замораживать ввод
      // В идеале хорошо бы сделать debounce, но с ним возникли проблемы, с точной причиной которых пока не разобрался
      // Отписываться смысла нет
      window.requestAnimationFrame(() => {
         this.props.dispatch(action);
      });
   }

   private loadFromStore() {
      const { formRecord } = this.props;
      if (formRecord?.state) {
         // https://st.yandex-team.ru/DEPLOY-4119
         // Почему-то форма сбрасывает errors и touched, если применить синхронно. Откладываю на чуть-чуть.
         // Отписываться смысла нет
         window.requestAnimationFrame(() => this.props.formik.setFormikState(formRecord.state as FormikState<any>));
      }
   }

   private removeFromStore(formId: string) {
      this.setState({ lastChange: null });

      if (this.props.formRecord) {
         this.dispatch(formStatesSlice.actions.remove(formId));
      }
   }

   private saveToStore<T>(formId: string, form: FormikProps<T>) {
      const lastChange = new Date();
      const record: FormStoreRecord = { id: formId, lastChange, state: extractFormikState(form) };

      this.setState({ lastChange });

      this.dispatch(formStatesSlice.actions.add(record));
   }

   private updateForm<T>(formId: string, form: FormikProps<T>, previousForm: FormikProps<T>) {
      const pristine = isEqual(form.values, form.initialValues);
      if (pristine) {
         this.removeFromStore(formId);
      } else {
         // Please, do not delete comments
         // import diff from 'jest-diff';
         // console.log(diff(form.values, form.initialValues));
         this.saveToStore(formId, form);
      }

      this.props.onChange?.(formId, form.values, previousForm.values);
   }
}

const mapStateToProps = (state: RootState, { formId }: Props) => ({
   formRecord: selectCurrentFormRecord(state, formId),
});

export const FormLocalState = connectFormik<Props>(connectRedux(mapStateToProps)(FormLocalStateImpl));
