import { createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';
import { FormikState } from 'formik';

import type { RootState } from '../../../../redux';

export type ReducedFormikState = Omit<FormikState<any>, 'isValidating' | 'isSubmitting' | 'status' | 'submitCount'>;

export interface FormStoreRecord {
   id: string;
   lastChange: Date | null;
   state: ReducedFormikState;
   updateType?: 'targetValue';
}

const SLICE_NAME = 'formStates';

type Prefix = string;

const initialState: Record<string, FormStoreRecord> = {};

export const formStatesSlice = createSlice({
   name: SLICE_NAME,
   initialState,
   reducers: {
      add(state, action: PayloadAction<FormStoreRecord>) {
         state[action.payload.id] = action.payload;
      },
      addMany(state, action: PayloadAction<FormStoreRecord[]>) {
         for (const record of action.payload) {
            state[record.id] = record;
         }
      },
      remove(state, action: PayloadAction<FormStoreRecord['id']>) {
         delete state[action.payload];
      },
      clearByPrefix(state, action: PayloadAction<Prefix>) {
         Object.keys(state)
            .filter(k => k.startsWith(action.payload))
            .forEach(k => delete state[k]);
      },
   },
});

/**
 * Map всех сохраненных узлов формы сущности от root
 */
export const selectFormStates: (state: RootState) => Map<string, FormStoreRecord> = createSelector(
   (state: RootState) => state.formStates,
   states => new Map(Object.keys(states).map(k => [k, states[k]])),
);

/**
 * Признак того, что в форме сущности есть несохраненные изменения
 */
export const selectFormsHasUnsavedChanges: (state: RootState, prefix: string) => boolean = createSelector(
   (state: RootState, prefix: string) => prefix,
   (state: RootState) => state.formStates,
   (prefix, states) => Object.keys(states).some(k => k.startsWith(prefix)),
);

export const selectCurrentFormRecord: (
   state: RootState,
   formId: string,
) => FormStoreRecord | undefined = createSelector(
   (state: RootState, formId: string) => formId,
   (state: RootState) => state.formStates,
   (formId, states) => states[formId],
);

/**
 * Текущее состояние узла формы сущности
 */
export const selectCurrentFormState: (
   state: RootState,
   formId: string,
) => ReducedFormikState | undefined = createSelector(selectCurrentFormRecord, record => record?.state);
