import { createSelector, createSlice, PayloadAction } from '@reduxjs/toolkit';

import type { RootState } from '../../../../redux';
import { AnySubForm } from '../../../huge-form';

const SLICE_NAME = 'nodeStates';

type Prefix = string;

interface NodeStoreRecord {
   level: string;
   mode: 'added' | 'removed';
   parentForms?: string[];
}

const initialState: Record<string, NodeStoreRecord> = {};

interface NodePayload {
   id: string;
   level: string;
   parentForm: AnySubForm;
}

type RemoveNodePayload = Omit<NodePayload, 'parentForm'>;

export const nodeStateSlice = createSlice({
   name: SLICE_NAME,
   initialState,
   reducers: {
      addSeveral(state, action: PayloadAction<NodePayload[]>) {
         for (const { id, level, parentForm } of action.payload) {
            state[id] = {
               level,
               mode: 'added',
               parentForms: [parentForm, ...parentForm.parentForms].map(f => f.id),
            };
         }
      },
      restore(state, action: PayloadAction<NodePayload['id']>) {
         const id = action.payload;
         if (state[id]?.mode === 'removed') {
            delete state[id];
         }
      },
      remove(state, action: PayloadAction<RemoveNodePayload>) {
         const { id, level } = action.payload;
         if (state[id]) {
            delete state[id];
         } else {
            state[id] = {
               level,
               mode: 'removed',
            };
         }
      },
      clearByPrefix(state, action: PayloadAction<Prefix>) {
         const prefix = action.payload;
         const keysForRemove = Object.keys(state).filter(key => key.startsWith(prefix));
         for (const key of keysForRemove) {
            delete state[key];
         }
      },
   },
});

export const selectNodeList: (state: RootState, prefix: string) => Map<string, NodeStoreRecord> = createSelector(
   (state: RootState, prefix: string) => prefix,
   (state: RootState) => state.nodeStates,
   (prefix, nodeStates) =>
      new Map(
         Object.keys(nodeStates)
            .filter(key => key.startsWith(prefix))
            .map(key => [key, nodeStates[key]]),
      ),
);

export const selectNodesHasUnsavedChanges: (state: RootState, prefix: string) => boolean = createSelector(
   selectNodeList,
   nodeList => nodeList.size > 0,
);
