import React, { ReactNode } from 'react';

import { IconDefinition } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { formatNumber } from '@yandex-infracloud-ui/libs';

import { PromoTooltip } from '../../../common';

import classes from './FormTab.module.css';

export interface FormTabProps {
   /**
    * Признак "вкладки"
    *
    * В production сборке при итерации через React.children ломаются displayName.
    * Это из-за того, что для каждого из child генерируется обертка React.Fragment.
    *
    * Один из вариантов решения - зафорсить значение некой пропсы, чтобы был признак, по которому отсеивать детей.
    *
    * Обычно FormTabs должны использовать FormTab в качестве элемента, но если хочется свой компонент для вкладки,
    * то он тоже должен у себя прописать этот defaultProps
    *
    * @see inspired by https://mparavano.medium.com/find-filter-react-children-by-type-d9799fb78292
    */
   __ROLE?: 'FormTab';
   count?: number;
   disabled?: boolean;
   emptyIcon?: IconDefinition;
   fillIcon?: IconDefinition;
   hint?: string;
   id: string;
   isEmpty?: boolean;
   isRequired?: boolean;
   title: string;
   promo?: Promo;
   children: ReactNode;
}

interface Promo {
   key: string;
   message: string;
}

export const FormTab: React.FC<FormTabProps> = React.memo(
   ({ title, isEmpty, emptyIcon, fillIcon, count, isRequired, promo }) => {
      let icon: ReactNode = null;
      if (isEmpty === true && emptyIcon) {
         icon = <FontAwesomeIcon icon={emptyIcon} fixedWidth={true} />;
      } else if (isEmpty === false && fillIcon) {
         icon = <FontAwesomeIcon icon={fillIcon} fixedWidth={true} />;
      }

      const TabContent = (
         <>
            {icon} {title}
            {count !== undefined ? <span className={classes.count}> {formatNumber(count)}</span> : null}
            {isRequired ? <span className={classes.required}>*</span> : null}
         </>
      );

      return (
         <>
            {promo ? (
               <PromoTooltip storageKey={promo.key} message={promo.message} title={title}>
                  {TabContent}
               </PromoTooltip>
            ) : (
               TabContent
            )}
         </>
      );
   },
);

FormTab.defaultProps = {
   __ROLE: 'FormTab',
};

FormTab.displayName = 'FormTab';
