import { faCopy, faTrash, faTrashUndo } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { classNames, FixedFormik } from '@yandex-infracloud-ui/libs';
import { Form } from 'formik';
import React, { useCallback, useContext, useMemo } from 'react';
import { useSelector } from 'react-redux';
import { RouteComponentProps } from 'react-router-dom';

import { LegoButton } from '../../../_lego';
import { Entity } from '../../../redux/models';
import { useLocationChangeRender } from '../../../utils';
import { FormLocalState, FormRevertButton, selectFormStates } from '../../forms';
import { DevForm, DevJson } from '../../lib';
import {
   AnySubForm,
   FormLevelConfig,
   getValidationContextForForm,
   HugeFormContext,
   SubForm,
   ValidationContext,
} from '../models';

import classes from './FormHOC.module.css';

interface FormHocParams<FP extends Entity, E extends Entity, SE extends Entity> {
   levelConfig: FormLevelConfig<FP, E, SE>;

   onChange(formId: string, newValues: any, oldValues: any): void;

   onClone(formId: string): void;

   onRemove(formId: string): void;

   onRestore(formId: string): void;

   onRevert(formId: string): void;

   setRevertHandler(h: () => void): void;
}

/**
 * HOC для формы узла (Рендерится роутером)
 *
 * FP - FormParams
 * E - Entity
 * SE - SubEntity
 */
export function formHOC<FP extends Entity, E extends Entity, SE extends Entity>({
   levelConfig: { component: FormComponent, renderTitle, validationSchema },
   onChange,
   onClone,
   onRemove,
   onRestore,
   onRevert,
   setRevertHandler,
}: FormHocParams<FP, E, SE>): React.FC<RouteComponentProps> {
   const stubSubmit = () => undefined;

   const Wrapped: React.FC<RouteComponentProps> = ({ location }) => {
      const formId = location.pathname;
      const context = useContext(HugeFormContext);
      const subForm: SubForm<FP, E, SE> | undefined = context.formMap.get(formId);

      const formStates = useSelector(selectFormStates);

      const handleClone = useCallback(() => onClone(formId), [formId]);

      const handleRemove = useCallback(() => onRemove(formId), [formId]);

      const handleRestore = useCallback(() => onRestore(formId), [formId]);

      const validationContext: ValidationContext = useMemo(() => {
         if (!subForm) {
            return {
               form: subForm,

               childrenForms: [],
               cousinForms: [],
               siblingForms: [],
               parentForms: [],

               siblingIds: new Set(),
               siblingAndCousinIds: new Set(),
            };
         }

         const allForms = Array.from(context.formMap.values());

         return getValidationContextForForm(subForm, allForms, formStates);
      }, [context.formMap, formStates, subForm]);

      const visible = useLocationChangeRender(location.pathname);
      if (!visible) {
         return null;
      }

      if (!subForm) {
         return (
            <div>
               Form <code>{formId}</code> not found!
            </div>
         );
      }

      return (
         <FixedFormik
            initialValues={subForm.formParams}
            enableReinitialize={true}
            onSubmit={stubSubmit}
            validationSchema={validationSchema}
            validationContext={validationContext}
         >
            {formik => {
               setRevertHandler(formik.resetForm);
               // formik.setValues(subForm.formParams);

               return (
                  <Form
                     className={classNames(classes.form, { [classes.readonly]: context.readonly || subForm.isRemoved })}
                     data-e2e={`SubForm:${subForm.levelConfig.id}`}
                  >
                     <header className={classes.header} data-e2e={'SubForm:Header'}>
                        <h2 data-e2e={'SubForm:Title'}>{renderTitle(subForm.value)}</h2>

                        <div className={classes.actions} data-e2e={'SubForm:Actions'}>
                           <FormLocalState formId={formId} onRevert={onRevert} onChange={onChange}>
                              {({ lastChange, revert }) => (
                                 <FormRevertButton
                                    buttonTitle={'Revert to spec values'}
                                    className={classes.changes}
                                    lastChange={lastChange}
                                    onClick={revert}
                                    readonly={subForm?.isAdded ?? false}
                                 />
                              )}
                           </FormLocalState>

                           {!context.readonly && subForm.levelConfig.level > 0 ? (
                              subForm.isRemoved ? (
                                 subForm.parentForms[0].isRemoved ? null : (
                                    <LegoButton
                                       title={'Restore'}
                                       disabled={context.disabled}
                                       onClick={handleRestore}
                                       controlAttrs={{ 'data-e2e': 'SubForm:RestoreButton' }}
                                    >
                                       <FontAwesomeIcon icon={faTrashUndo} fixedWidth={true} />
                                    </LegoButton>
                                 )
                              ) : (
                                 <>
                                    <LegoButton
                                       title={'Clone'}
                                       disabled={context.disabled}
                                       onClick={handleClone}
                                       controlAttrs={{ 'data-e2e': 'SubForm:CloneButton' }}
                                    >
                                       <FontAwesomeIcon icon={faCopy} fixedWidth={true} />
                                    </LegoButton>

                                    <LegoButton
                                       title={'Remove'}
                                       disabled={context.disabled}
                                       onClick={handleRemove}
                                       controlAttrs={{ 'data-e2e': 'SubForm:RemoveButton' }}
                                    >
                                       <FontAwesomeIcon icon={faTrash} fixedWidth={true} />
                                    </LegoButton>
                                 </>
                              )
                           ) : null}
                        </div>
                     </header>

                     {/* <DevJson summary={'Spec value'}>{'TODO'}</DevJson> */}

                     <DevJson summary={'subForm'}>{excludeBoringFields(subForm)}</DevJson>

                     <DevJson summary={'Model value'}>{subForm?.value}</DevJson>

                     <DevForm />

                     <FormComponent
                        disabled={context.disabled || formik.isSubmitting}
                        form={subForm}
                        formik={formik}
                        isRootEntityNew={context.isNew}
                        readonly={Boolean(context.readonly || subForm.isRemoved)}
                     />
                  </Form>
               );
            }}
         </FixedFormik>
      );
   };

   if (FormComponent.displayName) {
      Wrapped.displayName = `formHOC(${FormComponent.displayName})`;
   }

   return Wrapped;
}

function excludeBoringFields(subForm: AnySubForm): any {
   return {
      ...subForm,
      parentForms: subForm.parentForms.map(f => f.id),
      levelConfig: {
         ...subForm.levelConfig,
         validationSchema: '***',
         icon: '***',
         iconChanged: '***',
      },
      value: '***',
   };
}
