import { faBuilding, faObjectGroup, faUser } from '@fortawesome/free-solid-svg-icons';
import { boolean } from '@storybook/addon-knobs';
import { Meta } from '@storybook/react';
import { InputField2 } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useRef, useState } from 'react';
import { array, object, string } from 'yup';

import { LegoButton } from '../../../_lego';
import { routerDecorator } from '../../../stories';
import { DevJson } from '../../lib';
import { AnyFormLevelConfig, FormLevelConfig, HugeFormRef } from '../models';

import { HugeForm } from './HugeForm';

const JustShowFormConfig = () => (
   <>
      <InputField2 name={'id'} label={'ID'} readonly={true} />

      <InputField2 name={'name'} label={'Name'} />
   </>
);

interface Employee {
   id: string;
   name: string;
}

const employeeValidationSchema = object<Employee>({
   id: string().required(),
   name: string().required(),
});

const employeeLevelConfig: FormLevelConfig<Employee, Employee, any> = {
   component: JustShowFormConfig,
   getEmptyValue: () => ({ id: '', name: '' }),
   formParamsToValue: v => v,
   getChildren: () => [],
   icon: faUser,
   iconChanged: faUser,
   iconColor: 'red',
   id: 'employee',
   level: 2,
   name: 'Employee',
   renderTitle: () => 'Employee title',
   routePath: id => `/e_${id}`,
   validationSchema: employeeValidationSchema,
   valueToFormParams: v => v,
};

interface Department {
   id: string;
   name: string;
   employees: Employee[];
}

const departmentValidationSchema = object<Department>({
   id: string().required(),
   name: string().required(),
   employees: array().of(employeeValidationSchema),
});

const departmentLevelConfig: FormLevelConfig<Omit<Department, 'employees'>, Department, Employee> = {
   component: JustShowFormConfig,
   getEmptyValue: () => ({ id: '', name: '', employees: [] }),
   formParamsToValue: v => ({ id: v.id, name: v.name, employees: [] }),
   getChildren: d => d.employees,
   icon: faObjectGroup,
   iconChanged: faObjectGroup,
   iconColor: 'blue',
   id: 'department',
   level: 1,
   name: 'Department',
   renderTitle: () => 'Department title',
   routePath: id => `/d_${id}`,
   validationSchema: departmentValidationSchema,
   valueToFormParams: v => ({ id: v.id, name: v.name }),
};

interface Company {
   id: string;
   name: string;
   departments: Department[];
}

const companyValidationSchema = object<Company>({
   id: string().required(),
   name: string().required(),
   departments: array().of(departmentValidationSchema),
});

const companyLevelConfig: FormLevelConfig<Omit<Company, 'departments'>, Company, Department> = {
   component: JustShowFormConfig,
   getEmptyValue: () => ({ id: '', name: '', departments: [] }),
   formParamsToValue: v => ({ id: v.id, name: v.name, departments: [] }),
   getChildren: c => c.departments,
   icon: faBuilding,
   iconChanged: faBuilding,
   iconColor: 'green',
   id: 'company',
   level: 0,
   name: 'Company',
   renderTitle: () => 'Company title',
   routePath: () => '',
   validationSchema: companyValidationSchema,
   valueToFormParams: v => ({ id: v.id, name: v.name }),
};

const levelConfigs: AnyFormLevelConfig[] = [companyLevelConfig, departmentLevelConfig, employeeLevelConfig];

export const Regular = () => {
   const formRef = useRef<HugeFormRef<Company>>(null);

   const [value, setValue] = useState({
      id: 'yandex',
      name: 'Яндекс',
      departments: [
         {
            id: 'search',
            name: 'Поиск',
            employees: [
               { id: 'khoden', name: 'Березин Артём' },
               { id: 'ivan', name: 'Иванов Иван' },
            ],
         },
      ],
   } as Company);
   const [newValue, setNewValue] = useState<any>();

   const handleGetValue = useCallback(() => {
      if (formRef.current) {
         setNewValue(formRef.current.getValue());
      }
   }, []);

   const handleEditValue = useCallback(() => {
      setValue({
         ...value,
         name: 'Mail.ru',
      });
   }, [value]);

   return (
      <>
         <LegoButton onClick={handleEditValue}>edit value externally</LegoButton>

         <HugeForm
            disabled={boolean('disabled', false)}
            isNew={boolean('isNew', false)}
            levelConfigs={levelConfigs}
            readonly={boolean('readonly', false)}
            value={value}
            ref={formRef}
            rootRoutePath={() => '/storibook-huge-form'}
         />

         <LegoButton onClick={handleGetValue}>get value from form</LegoButton>
         <DevJson open={true} summary={'Value from form'}>
            {newValue}
         </DevJson>
      </>
   );
};

export default {
   decorators: [routerDecorator],
   title: 'components/huge-form/HugeForm',
} as Meta;
