import { getSetDifference, isEqual, SwitchRoutes } from '@yandex-infracloud-ui/libs';
import React, {
   forwardRef,
   ForwardRefRenderFunction,
   useCallback,
   useEffect,
   useImperativeHandle,
   useMemo,
   useRef,
   useState,
} from 'react';
import { useHistory } from 'react-router';
import { Spin } from '@yandex-cloud/uikit';

import { AppRouteProps } from '../../../models';
import { Entity } from '../../../redux/models';
import { noop, useThrowIfChanged } from '../../../utils';
import { useFormsActions } from '../hooks/useFormsActions';
import { useFormsFromValue } from '../hooks/useFormsFromValue';
import { useValueFromForms } from '../hooks/useValueFromForms';
import { useTargetValueState } from '../hooks/useTargetValueState';

import {
   AnyFormLevelConfig,
   AnySubForm,
   AnyLevelParamsMap,
   FormChangeListener,
   HugeFormContext,
   HugeFormContextContent,
   HugeFormRef,
   sortForms,
} from '../models';
import { SideTree } from '../SideTree/SideTree';
import { formHOC } from './FormHOC';
import classes from './HugeForm.module.css';

const isDefaultEqualForms = (formA: AnySubForm, formB: AnySubForm) => isEqual(formA.formParams, formB.formParams);
interface Props {
   disabled: boolean;

   /**
    * внешние обработчики событий изменения полей
    */
   formChangeListeners?: FormChangeListener[];
   isNew: boolean;
   levelConfigs: AnyFormLevelConfig[];
   payload?: Record<string, any>;
   currentEmptyParams?: AnyLevelParamsMap;
   readonly: boolean;
   value: Entity;

   /**
    * начальное значение формы, если задано, то value будет получено так, как если бы форму заранее редактировали
    */
   initialValue?: Entity;

   /**
    * кастомное сравнение двух форм, по умолчанию isEqual(formA.formParams, formB.formParams)
    */
   isEqualForms?(formA: AnySubForm, formB: AnySubForm): boolean;

   onFormRevert?(formId: string): void;

   onNodeClone?(levelId: string, originalInitialId: string, newId: string): void;

   onNodeListChange?(): void;

   rootRoutePath(rootEntityId: string): string;
}

const HugeFormInternal: ForwardRefRenderFunction<HugeFormRef<any>, Props> = (
   {
      disabled,
      formChangeListeners,
      isNew,
      levelConfigs,
      onFormRevert,
      onNodeClone,
      onNodeListChange,
      payload = {},
      currentEmptyParams = new Map(),
      readonly,
      rootRoutePath,
      value,
      initialValue,
      isEqualForms = isDefaultEqualForms,
   },
   ref,
) => {
   const history = useHistory();

   const useInitialValue = Boolean(initialValue) && !isNew;
   const preparedInitialValue = initialValue && useInitialValue ? initialValue : value;

   const initialForms = useFormsFromValue(preparedInitialValue, levelConfigs, isNew, rootRoutePath);
   const targetForms = useFormsFromValue(value, levelConfigs, isNew, rootRoutePath);
   const isEqualRoot = initialForms[0]?.id === targetForms[0]?.id;

   const initialFormsMap = useMemo(() => new Map(initialForms.map(v => [v.id, v])), [initialForms]);
   const targetFormsMap = useMemo(() => new Map(targetForms.map(v => [v.id, v])), [targetForms]);

   const preparedForms = useMemo(() => {
      if (!useInitialValue) {
         return targetForms;
      }
      const { unchanged: unchangedFormIds, removed: removedFormIds } = getSetDifference(
         new Set(initialFormsMap.keys()),
         new Set(targetFormsMap.keys()),
      );
      const commonForms = Array.from(unchangedFormIds).map(id => targetFormsMap.get(id)!);
      const removingForms = Array.from(removedFormIds).map(id => initialFormsMap.get(id)!);
      const resultForms = [...commonForms, ...removingForms];
      sortForms(resultForms);
      return resultForms;
   }, [initialFormsMap, targetForms, targetFormsMap, useInitialValue]);

   const [forms, setForms] = useState(preparedForms);
   useEffect(() => {
      setForms(preparedForms);
   }, [preparedForms]);

   const currentFormRevertRef = useRef<() => void>();

   const updateForms = useCallback(
      (f: AnySubForm[], redirectTo?: string) => {
         setForms(f);

         if (redirectTo) {
            history.push(redirectTo);
         }
      },
      [history],
   );

   const resetCurrentForm = useCallback(() => {
      if (currentFormRevertRef.current) {
         currentFormRevertRef.current();
      }

      updateForms(initialForms);
   }, [initialForms, updateForms]);

   const levelConfigMap = useMemo(() => new Map(levelConfigs.map(c => [c.level, c])), [levelConfigs]);
   const formMap = useMemo(() => new Map(forms.map(f => [f.id, f])), [forms]);

   const context = useMemo<HugeFormContextContent>(
      () => ({
         disabled,
         formMap,
         isNew,
         levelConfigMap,
         payload,
         currentEmptyParams,
         readonly,
      }),
      [disabled, formMap, isNew, levelConfigMap, payload, currentEmptyParams, readonly],
   );

   const {
      addForm,
      cloneForm,
      removeForm,
      restoreForm,
      onChangeForm,
      insertForms,
      removeForms,
      handleChangeForm,
   } = useFormsActions({
      context,
      forms,
      setForms: updateForms,
      onNodeListChange: onNodeListChange ?? noop,
      onNodeClone: onNodeClone ?? noop,
      formChangeListeners,
   });

   const { isLoadTarget } = useTargetValueState({
      initialValue: preparedInitialValue,
      value,
      initialFormsMap,
      targetFormsMap,
      isEqualRoot,
      insertForms,
      removeForms,
      isEqualForms,
      handleChangeForm,
   });

   const routes = useMemo(() => {
      let previousPath = '';

      return levelConfigs.map(levelConfig => {
         // side effect
         previousPath =
            levelConfig.level === 0
               ? rootRoutePath(`:${levelConfig.id}`)
               : `${previousPath}${levelConfig.routePath(`:${levelConfig.id}`)}`;

         return {
            exact: true,
            path: previousPath,
            component: formHOC({
               levelConfig,
               onChange: onChangeForm,
               onClone: cloneForm,
               onRemove: removeForm,
               onRestore: restoreForm,
               onRevert: onFormRevert ?? noop,
               setRevertHandler: fn => {
                  currentFormRevertRef.current = fn;
               },
            }),
         } as AppRouteProps;
      });
   }, [levelConfigs, rootRoutePath, onChangeForm, cloneForm, removeForm, restoreForm, onFormRevert]);

   useThrowIfChanged(formChangeListeners, 'formChangeListeners recreate');

   // console.log({ routes, forms }); // Don't remove the line, please!

   // ref logic
   const getValue = useValueFromForms(forms, levelConfigs);
   useImperativeHandle(
      ref,
      () => ({
         getForms: () => forms,
         getValue,
         resetCurrentForm,
      }),
      [forms, getValue, resetCurrentForm],
   );

   // render
   return (
      <HugeFormContext.Provider value={context}>
         <div className={classes.wrapper} data-e2e={'HugeForm'}>
            {isLoadTarget ? (
               <Spin />
            ) : (
               <>
                  <SideTree forms={forms} onFormAdd={addForm} readonly={readonly} />

                  <div className={classes.content} data-e2e={'HugeForm:Content'}>
                     <SwitchRoutes doNotAddPrefix={true}>{routes}</SwitchRoutes>
                  </div>
               </>
            )}
         </div>
      </HugeFormContext.Provider>
   );
};

HugeFormInternal.displayName = 'HugeFormInternal';

export const HugeForm = forwardRef(HugeFormInternal);

HugeForm.displayName = 'HugeForm';
