import { formatNumber, hasIncluded, plural, removeSetItem, styleHelpers } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useContext, useState } from 'react';
import { useSelector } from 'react-redux';

import { selectFormStates } from '../../forms';
import { FastSearchInput } from '../../lib';

import { AnySubForm, HugeFormContext } from '../models';
import classes from './SideTree.module.css';
import { SideTreeItem } from './SideTreeItem';

interface Props {
   forms: AnySubForm[];
   readonly: boolean;

   onFormAdd(parentFormId: string): void;
}

export const SideTree: React.FC<Props> = React.memo(({ forms, onFormAdd, readonly }) => {
   const [query, setQuery] = useState('');
   const [collapsed, setCollapsed] = useState(new Set<string>());
   const { levelConfigMap } = useContext(HugeFormContext);

   const formStates = useSelector(selectFormStates);

   const filterForms = useCallback(
      (form: AnySubForm, i: number) => {
         if (i === 0 || !query || hasIncluded(query, form.formParams.id)) {
            return true;
         }

         // Если есть дети, которые попадают под запрос, тогда родителя тоже не надо прятать
         let cursor = i + 1;
         let nextItem = forms[cursor];
         while (nextItem && nextItem.levelConfig.level > form.levelConfig.level) {
            if (hasIncluded(query, nextItem.value.id)) {
               return true;
            }
            cursor += 1;
            nextItem = forms[cursor];
         }

         return false;
      },
      [forms, query],
   );

   const skipCollapsed = useCallback(
      (form: AnySubForm, i: number) => {
         if (i === 0 || collapsed.size === 0) {
            return true;
         }

         return !form.parentForms.some(p => collapsed.has(p.id));
      },
      [collapsed],
   );

   const filteredForms = forms.filter(filterForms).filter(skipCollapsed);

   const handleToggleCollapsed = useCallback((formId: string) => {
      setCollapsed(c => {
         // was collapsed
         if (c.has(formId)) {
            // just remove
            return removeSetItem(c, formId);
         }

         // was expanded
         return new Set([...Array.from(c).filter(id => !id.startsWith(formId)), formId]);
      });
   }, []);

   return (
      <aside className={classes.aside} data-e2e={'side-tree'}>
         <nav data-e2e={'SideTree:Nav'}>
            {forms.length > 10 ? (
               <div data-e2e={'SideTree:Filter'}>
                  <FastSearchInput
                     className={classes.search}
                     value={query}
                     onChange={setQuery}
                     forceVisible={true}
                     placeholder={'Fast filter nodes...'}
                     name={'side-tree-query'}
                  />
               </div>
            ) : null}

            <ul className={styleHelpers.unstyledList} data-e2e={'SideTree:ItemList'}>
               {filteredForms.map(form => {
                  const formState = formStates.get(form.id);

                  const errorCount = Object.keys(formState?.state.errors ?? {}).filter(fieldWithError =>
                     Boolean(formState?.state.touched[fieldWithError]),
                  ).length;

                  return (
                     <SideTreeItem
                        key={form.id}
                        changed={Boolean(formState?.lastChange)}
                        childConfig={levelConfigMap.get(form.levelConfig.level + 1)}
                        collapsed={collapsed.has(form.id)}
                        form={form}
                        onAddChild={onFormAdd}
                        onToggleCollapsed={handleToggleCollapsed}
                        query={query}
                        readonly={readonly}
                     >
                        {errorCount > 0 ? (
                           <div className={classes.error} data-e2e={'SideTree:HasError'}>
                              Has {plural(errorCount, 'an error', `${formatNumber(errorCount)} errors`)}
                           </div>
                        ) : null}
                     </SideTreeItem>
                  );
               })}
            </ul>

            {query && filteredForms.length === 1 ? (
               <p className={classes.empty} data-e2e={'SideTree:NoMatchedNodes'}>
                  No matched nodes
               </p>
            ) : null}

            {forms.length === 1 ? (
               <p className={classes.empty} data-e2e={'SideTree:EmptyList'}>
                  Empty list
               </p>
            ) : null}
         </nav>
      </aside>
   );
});

SideTree.displayName = 'SideTree';
