import { faChevronDown, faChevronUp, faPlus } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { classNames, forHumanCapitalized, FormButton, HighlightedString } from '@yandex-infracloud-ui/libs';
import React, { SyntheticEvent, useCallback } from 'react';
import { useSelector } from 'react-redux';
import { NavLink } from 'react-router-dom';

import type { RootState } from '../../../redux';
import { createE2eSelector } from '../../../utils';

import { selectCurrentFormState } from '../../forms';

import { AnyFormLevelConfig, AnySubForm } from '../models';
import classes from './SideTreeItem.module.css';

interface Props {
   changed: boolean;
   childConfig: AnyFormLevelConfig | undefined;
   collapsed: boolean;
   form: AnySubForm;
   query: string;
   readonly: boolean;

   onAddChild(formId: string): void;

   onToggleCollapsed(formId: string): void;
}

export const SideTreeItem: React.FC<Props> = ({
   changed,
   childConfig,
   children,
   collapsed,
   form,
   onAddChild,
   onToggleCollapsed,
   query,
   readonly,
}) => {
   const canCollapse = form.hasChildren && form.levelConfig.level > 0;
   const childrenName = childConfig ? childConfig.name ?? forHumanCapitalized(childConfig.id) : '';
   const currentFormValuesSelector = useCallback((s: RootState) => selectCurrentFormState(s, form.id)?.values, [
      form.id,
   ]);
   const currentFormValues = useSelector(currentFormValuesSelector);

   const handleToggleCollapsed = useCallback(
      (e: SyntheticEvent) => {
         e.preventDefault();
         e.stopPropagation();

         onToggleCollapsed(form.id);
      },
      [form.id, onToggleCollapsed],
   );

   const handleAddChild = useCallback(
      (e: SyntheticEvent) => {
         e.preventDefault();
         e.stopPropagation();

         onAddChild(form.id);
      },
      [form.id, onAddChild],
   );

   const handleClick = useCallback(() => {
      window.scrollTo({ top: 0 });
   }, []);

   const touched = form.isAdded || changed || form.isRemoved;

   const name = currentFormValues?.id ?? form.value.id;

   return (
      <li
         data-e2e={`SideTreeItem:${form.levelConfig.id}`}
         className={classNames({
            [classes.added]: form.isAdded,
            [classes.removed]: form.isRemoved,
         })}
      >
         <NavLink
            activeClassName={classes.active}
            className={classes.link}
            exact={true}
            style={
               {
                  '--levelPadding': `calc(2 * ${Math.max(form.levelConfig.level - 1, 0)} * var(--shift))`,
               } as any
            }
            to={form.id}
            onClick={handleClick}
            data-e2e={'SideTreeItem:Link'}
         >
            <div>
               <div className={classes.firstRow} data-e2e={'SideTreeItem:firstRow'}>
                  <FontAwesomeIcon
                     className={classNames(classes.icon, { [classes.canCollapse]: canCollapse })}
                     color={form.levelConfig.iconColor}
                     icon={touched ? form.levelConfig.iconChanged : form.levelConfig.icon}
                     fixedWidth={true}
                     data-e2e={'SideTreeItem:Icon'}
                  />

                  <span
                     onDoubleClick={handleToggleCollapsed}
                     className={classNames(classes.name, {
                        [classes.touched]: touched,
                        [classes.changed]: !form.isAdded && changed,
                     })}
                     title={
                        `value.id: ${form.value.id}` // TODO remove
                     }
                     data-e2e={'SideTreeItem:Title'}
                  >
                     {name ? <HighlightedString query={query} content={name} /> : `[empty]`}
                  </span>

                  {canCollapse ? (
                     <FontAwesomeIcon
                        className={classes.expandIcon}
                        icon={collapsed ? faChevronUp : faChevronDown}
                        fixedWidth={true}
                        onClick={handleToggleCollapsed}
                        data-e2e={`SideTreeItem:${collapsed ? 'collapsed' : 'expanded'}Icon`}
                     />
                  ) : null}

                  {form.isAdded ? (
                     <span className={classes.mark} data-e2e={'SideTreeItem:isAdded'}>
                        (added)
                     </span>
                  ) : null}
                  {form.isRemoved ? (
                     <span className={classes.mark} data-e2e={'SideTreeItem:isRemoved'}>
                        (removed)
                     </span>
                  ) : null}
               </div>

               {children ? (
                  <div className={classes.secondRow} data-e2e={'SideTreeItem:secondRow'}>
                     {children}
                  </div>
               ) : null}
            </div>

            <div className={classes.actions} data-e2e={'SideTreeItem:Actions'}>
               {childConfig && !readonly && !form.isRemoved ? (
                  <FormButton
                     onClick={handleAddChild}
                     className={classes.lastAction}
                     title={`Add ${childrenName}`}
                     dataE2e={`Actions:${createE2eSelector(null, `Add ${childrenName} Button`)}`}
                  >
                     <FontAwesomeIcon icon={faPlus} fixedWidth={true} />
                     <FontAwesomeIcon icon={childConfig.icon} fixedWidth={true} />
                  </FormButton>
               ) : null}
            </div>
         </NavLink>
      </li>
   );
};

SideTreeItem.displayName = 'SideTreeItem';
