import { deepClone, isEqual, sortHandler, unique } from '@yandex-infracloud-ui/libs';
import { getIn } from 'formik';
import { useCallback } from 'react';
import { useDispatch, useStore } from 'react-redux';

import { Entity } from '../../../redux/models';
import { getSequentialId } from '../../../utils';
import { nodeStateSlice, selectCurrentFormState } from '../../forms';
import {
   AnySubForm,
   FormChangeListener,
   FormChangeParams,
   FormChangeType,
   getDirectChildForms,
   HugeFormContextContent,
   sortForms,
} from '../models';
import { formsFromValue } from './useFormsFromValue';
import { valueFromForms } from './useValueFromForms';

interface FormActionsParams {
   context: HugeFormContextContent;
   forms: AnySubForm[];
   setForms: (f: AnySubForm[], redirectTo?: string) => void;
   onNodeListChange: () => void;
   onNodeClone: (levelId: string, originalInitialId: string, newId: string) => void;
   formChangeListeners?: FormChangeListener[];
}

export function useFormsActions({
   context,
   forms,
   setForms,
   onNodeListChange,
   onNodeClone,
   formChangeListeners,
}: FormActionsParams) {
   const reduxStore = useStore();
   const dispatch = useDispatch();

   const handleChangeForm = useCallback(
      (form: AnySubForm, newFormValues: any, oldFormValues: any, type: FormChangeType) => {
         if (!formChangeListeners) {
            return;
         }

         const { levelConfig, parentForms } = form;
         const levelId = levelConfig.id;
         for (const { listenFields, onChange } of formChangeListeners) {
            if (listenFields.hasOwnProperty(levelId)) {
               const pathes = listenFields[levelId];
               for (const path of pathes) {
                  const newValue = getIn(newFormValues, path);
                  const oldValue = getIn(oldFormValues, path);
                  if (type !== FormChangeType.Edit || !isEqual(newValue, oldValue)) {
                     const params: FormChangeParams = {
                        levelId,
                        path,
                        newValue,
                        oldValue,
                        formValues: newFormValues,
                        parentForms,
                        type,
                     };
                     onChange(params);
                  }
               }
            }
         }
      },
      [formChangeListeners],
   );

   const getLevelConfigsFromLevel = useCallback(
      (level: number) =>
         Array.from(context.levelConfigMap.values())
            .filter(l => l.level > level)
            .sort((a, b) => sortHandler(a.level, b.level)),
      [context.levelConfigMap],
   );

   /**
    * добавление форм включая их дочерние формы
    */
   const addNewForms = useCallback(
      (formRecords: { value: Entity; parentForm: AnySubForm }[]) => {
         const allNewForms: AnySubForm[] = [];
         // хранилище узлов
         const allNewNodes: {
            id: string;
            level: string;
            parentForm: AnySubForm;
         }[] = [];

         for (const formRecord of formRecords) {
            const { value, parentForm } = formRecord;
            const siblings = parentForm.levelConfig.getChildren(parentForm.value);
            siblings.push(value);

            const descendantsLevelConfigs = getLevelConfigsFromLevel(parentForm.levelConfig.level);

            const newForms = formsFromValue({
               value,
               levelConfigs: descendantsLevelConfigs,
               parentForms: [parentForm, ...parentForm.parentForms],
               isNew: true,
            });

            allNewForms.push(...newForms);

            allNewNodes.push(
               ...newForms.map(f => ({
                  id: f.id,
                  level: f.levelConfig.id,
                  parentForm: f.parentForms[0],
               })),
            );
         }
         dispatch(nodeStateSlice.actions.addSeveral(allNewNodes));

         // список самих форм
         const allForms = [...forms, ...allNewForms];
         sortForms(allForms);
         setForms(allForms, allNewForms[0]?.id);

         onNodeListChange();

         // вычисления для новых полей
         for (const form of allNewForms) {
            handleChangeForm(form, form.formParams, form.formParams, FormChangeType.Add);
         }
      },
      [getLevelConfigsFromLevel, dispatch, forms, setForms, onNodeListChange, handleChangeForm],
   );

   const generateId = useCallback(
      (parentForm: AnySubForm, defaultId: string, isCloning: boolean) => {
         const siblingForms = getDirectChildForms(parentForm, forms);
         const oldSiblingIds = siblingForms.map(s => s.formParams.id);
         const newSiblingIds = siblingForms
            .map(s => selectCurrentFormState(reduxStore.getState(), s.id)?.values.id)
            .filter(Boolean);
         const siblingIds = unique([...oldSiblingIds, ...newSiblingIds]);

         return getSequentialId(defaultId, siblingIds, {
            suffix: isCloning ? 'clone' : '',
         });
      },
      [forms, reduxStore],
   );

   const replaceId = useCallback(
      (v: Entity, parentForm: AnySubForm, defaultId: string, isCloning: boolean) => {
         const uniqId = generateId(parentForm, defaultId, isCloning);
         v.id = uniqId;

         return uniqId;
      },
      [generateId],
   );

   const onChangeForm = useCallback(
      (formId: string, newFormValues: any, oldFormValues: any) => {
         const form = context.formMap.get(formId);
         if (form) {
            handleChangeForm(form, newFormValues, oldFormValues, FormChangeType.Edit);
         }
      },
      [context.formMap, handleChangeForm],
   );

   /**
    * добавление новой пустой формы
    */
   const addForm = useCallback(
      (parentFormId: string) => {
         const parentForm = context.formMap.get(parentFormId)!;
         const currentFormLevel = parentForm.levelConfig.level + 1;
         const levelConfig = context.levelConfigMap.get(currentFormLevel)!;

         const emptyParams = context.currentEmptyParams.get(currentFormLevel);
         const newValue = levelConfig.getEmptyValue(emptyParams);

         replaceId(newValue, parentForm, newValue.id, false);

         addNewForms([{ value: newValue, parentForm }]);
      },
      [addNewForms, context.currentEmptyParams, context.formMap, context.levelConfigMap, replaceId],
   );

   /**
    * вставка новых форм с готовыми значениями
    */
   const insertForms = useCallback(
      (formRecords: { parentFormId: string; value: Entity }[]) => {
         addNewForms(
            formRecords.map(({ parentFormId, value }) => ({ value, parentForm: context.formMap.get(parentFormId)! })),
         );
      },
      [addNewForms, context.formMap],
   );

   const cloneForm = useCallback(
      (formId: string) => {
         const form = context.formMap.get(formId)!;
         const parentForm = form.parentForms[0];

         const subForms = Array.from(context.formMap.values()).filter(f => f.id.startsWith(formId));
         const descendantsLevelConfigs = getLevelConfigsFromLevel(parentForm.levelConfig.level);
         // deepClone нужен, чтобы убрать иммутабельность из ответа reduxStore.getState()
         const newFormValue = deepClone(valueFromForms(subForms, descendantsLevelConfigs, reduxStore)!);
         if (form.levelConfig.clearOnClone) {
            form.levelConfig.clearOnClone(newFormValue);
         }

         const newId = replaceId(newFormValue, parentForm, newFormValue.id, true);
         addNewForms([{ value: newFormValue, parentForm }]);

         if (onNodeClone) {
            onNodeClone(form.levelConfig.id, form.formParams.id, newId);
         }
      },
      [context.formMap, getLevelConfigsFromLevel, reduxStore, replaceId, addNewForms, onNodeClone],
   );

   /**
    * множественное удаление форм
    */
   const removeForms = useCallback(
      (formIds: string[]) => {
         const formIdsSet = new Set(formIds);
         const newForms: AnySubForm[] = [];

         for (const form of forms) {
            const isNested = isFormNestedForIds(formIdsSet, form);
            if (formIdsSet.has(form.id) || isNested) {
               // до удаления запускаем обработчики событий
               handleChangeForm(form, form.formParams, form.formParams, FormChangeType.Remove);
               dispatch(nodeStateSlice.actions.remove({ id: form.id, level: form.levelConfig.id }));
               if (!form.isAdded) {
                  newForms.push({ ...form, isRemoved: true });
                  if (isNested) {
                     form.parentForms[0].isRemoved = true;
                  }
               }
            } else {
               newForms.push(form);
            }
         }

         for (const formId of formIds) {
            const currentForm = context.formMap.get(formId)!;
            const parentForm = currentForm.parentForms[0];
            const children = parentForm.levelConfig.getChildren(parentForm.value);

            // remove from model
            const i = children.findIndex(c => c.id === currentForm.value.id);
            children.splice(i, 1);
         }

         const currentForm = context.formMap.get(formIds[0])!;
         const parentForm = currentForm.parentForms[0];

         // set new forms
         setForms(newForms, parentForm?.id ?? undefined);

         onNodeListChange();
      },
      [context.formMap, setForms, onNodeListChange, forms, handleChangeForm, dispatch],
   );

   const removeForm = useCallback(
      (formId: string) => {
         removeForms([formId]);
      },
      [removeForms],
   );

   const restoreForm = useCallback(
      (formId: string) => {
         const newForms: AnySubForm[] = [];
         for (const form of forms) {
            // foreign forms keep untouched
            if (form.id === formId || isFormParent(formId, form)) {
               // до восстановления запускаем обработчики событий
               handleChangeForm(form, form.formParams, form.formParams, FormChangeType.Add);

               dispatch(nodeStateSlice.actions.restore(form.id));
               newForms.push({ ...form, isRemoved: false });
            } else {
               newForms.push(form);
            }
         }

         const currentForm = forms.find(f => f.id === formId)!;
         const parentForm = currentForm.parentForms[0];
         const children = parentForm.levelConfig.getChildren(parentForm.value);

         // return value to model
         children.push(currentForm.levelConfig.formParamsToValue(currentForm.formParams, currentForm.value));

         setForms(newForms);

         onNodeListChange();
      },
      [forms, setForms, onNodeListChange, handleChangeForm, dispatch],
   );

   return { addForm, cloneForm, removeForm, restoreForm, onChangeForm, insertForms, removeForms, handleChangeForm };
}

function isFormParent(parentFormId: string, form: AnySubForm): boolean {
   return form.parentForms.some(f => f.id === parentFormId);
}

function isFormNestedForIds(parentFormIds: Set<string>, form: AnySubForm): boolean {
   return form.parentForms.some(f => parentFormIds.has(f.id));
}
