import { isEmpty } from '@yandex-infracloud-ui/libs';
import { useMemo } from 'react';

import { Entity } from '../../../redux/models';
import { AnyFormLevelConfig, AnySubForm, sortForms } from '../models';

interface Props {
   value: Entity;
   levelConfigs: AnyFormLevelConfig[];
   parentForms?: AnySubForm[];
   isNew?: boolean;
   rootRoutePath?: (id: string) => string;
}

export function formsFromValue({ value, levelConfigs, parentForms, isNew = false, rootRoutePath }: Props) {
   const isAdded = isNew ?? undefined;
   const forms: AnySubForm[] = [];

   let previousLevelForms: AnySubForm[] | null = null;
   let first = true;

   for (const levelConfig of levelConfigs) {
      const { routePath, valueToFormParams, getChildren } = levelConfig;

      // _fullId is needed to distinguish nodes with the same ID in <Formik initialValue/>
      const getFormParams = (id: string, v: any) => ({ ...valueToFormParams(v), _fullId: id, initialId: v.id });

      if (first) {
         const children = getChildren(value);

         const parentForm = isEmpty(parentForms) ? undefined : parentForms[0];

         let id = `${parentForm?.id ?? ''}${routePath(value.id)}`;
         if (rootRoutePath) {
            id = rootRoutePath(value.id);
         }

         const rootForm: AnySubForm = {
            formParams: getFormParams(id, value),
            hasChildren: children.length > 0,
            id,
            isAdded,
            levelConfig,
            parentForms: parentForms ?? [],
            value,
         };

         forms.push(rootForm);
         previousLevelForms = [rootForm];
         first = false;
      } else {
         previousLevelForms = previousLevelForms!
            .map(parentForm =>
               parentForm.levelConfig.getChildren(parentForm.value).map(child => {
                  const id = `${parentForm.id}${routePath(child.id)}`;

                  return {
                     formParams: getFormParams(id, child),
                     hasChildren: getChildren(child).length > 0,
                     id,
                     isAdded,
                     levelConfig,
                     parentForms: [parentForm, ...parentForm.parentForms],
                     value: child,
                  } as AnySubForm;
               }),
            )
            .flat();
         forms.push(...previousLevelForms);
      }
   }

   // сортирую список форм по полному пути (выстраивая по иерархии)
   sortForms(forms);

   return forms;
}

export function useFormsFromValue(
   value: Entity,
   levelConfigs: AnyFormLevelConfig[],
   isNew: boolean,
   rootRoutePath: (id: string) => string,
) {
   return useMemo(
      () =>
         formsFromValue({
            value,
            levelConfigs,
            isNew,
            rootRoutePath,
         }),
      [levelConfigs, value, isNew, rootRoutePath],
   );
}
