import { getSetDifference } from '@yandex-infracloud-ui/libs';
import { useEffect, useState } from 'react';
import { useDispatch } from 'react-redux';
import { Entity } from '../../../redux/models';
import { formStatesSlice, FormStoreRecord } from '../../forms';
import { AnySubForm, FormChangeType } from '../models';

const getFormStoreRecord = (form: AnySubForm) => {
   const lastChange = new Date();
   const record: FormStoreRecord = {
      id: form.id,
      lastChange,
      state: { values: form.formParams, errors: {}, touched: {} },
      updateType: 'targetValue',
   };
   return record;
};

interface FormRecord {
   parentFormId: string;
   value: Entity;
}

interface UseTargetValueStateParams {
   /**
    * возможное начальное значение
    * */
   initialValue?: Entity;

   /**
    * актуальное значение
    * */
   value: Entity;

   /**
    * начальный набор форм
    * */
   initialFormsMap: Map<string, AnySubForm>;

   /**
    * целевой набор форм
    * */
   targetFormsMap: Map<string, AnySubForm>;

   /**
    * корень у начального и целевого набора форм совпадает
    */
   isEqualRoot: boolean;

   /**
    * массовая вставка форм
    */
   insertForms: (formRecords: FormRecord[]) => void;

   /**
    * массовое удаление форм
    */
   removeForms: (formIds: string[]) => void;

   /**
    * проверка эквивалентности форм на основе значимых полей
    */
   isEqualForms(formA: AnySubForm, formB: AnySubForm): boolean;

   /**
    * запуск обработчиков на изменение
    */
   handleChangeForm(form: AnySubForm, newFormValues: any, oldFormValues: any, type: FormChangeType): void;
}

interface UseTargetValueStateResult {
   /**
    * состояние формы в процессе приведения к целевому состоянию
    */
   isLoadTarget: boolean;
}

export function useTargetValueState({
   initialValue,
   value,
   initialFormsMap,
   targetFormsMap,
   isEqualRoot,
   insertForms,
   removeForms,
   isEqualForms,
   handleChangeForm,
}: UseTargetValueStateParams): UseTargetValueStateResult {
   const dispatch = useDispatch();

   // стартовое значение, запускает процесс приведения к целевому виду,
   // только если есть начальное значение и общий корень набора форм
   const startValue = initialValue && isEqualRoot ? initialValue : value;

   // ничего делать не требуется
   const alreadyTarget = startValue === value;

   const [usedTarget, setUsedTarget] = useState(alreadyTarget);

   useEffect(() => {
      // сброс при смене значений
      setUsedTarget(alreadyTarget);
   }, [alreadyTarget, value, initialValue]);

   useEffect(() => {
      if (usedTarget) {
         // целевое состояние достигнуто, ничего больше не делаем
         return;
      }

      const { removed, added, unchanged } = getSetDifference(
         new Set(initialFormsMap.keys()),
         new Set(targetFormsMap.keys()),
      );

      // набор форм для массовой вставки
      const addedFormRecords: FormRecord[] = [];
      for (const id of added) {
         const form = targetFormsMap.get(id)!;
         const parentFormId = form.parentForms[0].id;

         // все родительские формы уже добавляются вместе с потомками
         // поэтому потомков пропускаем
         if (!added.has(parentFormId)) {
            addedFormRecords.push({ parentFormId, value: form.value });
         }
      }
      if (addedFormRecords.length > 0) {
         insertForms(addedFormRecords);
      }

      // набор форм для массового удаления
      const removedIds: string[] = [];
      for (const id of removed) {
         const form = initialFormsMap.get(id)!;
         const parentFormId = form.parentForms[0].id;

         // все родительские формы уже удаляются вместе с потомками
         // поэтому потомков пропускаем
         if (!removed.has(parentFormId)) {
            removedIds.push(id);
         }
      }
      if (removedIds.length > 0) {
         removeForms(removedIds);
      }

      // формы, внутри которых есть изменения для отображения
      const changedIds = Array.from(unchanged).filter(id => {
         const formA = initialFormsMap.get(id)!;
         const formB = targetFormsMap.get(id)!;
         const existDiff = !isEqualForms(formA, formB);

         return existDiff;
      });

      if (changedIds.length > 0) {
         const formRecords = changedIds.map(id => targetFormsMap.get(id)!).map(getFormStoreRecord);
         dispatch(formStatesSlice.actions.addMany(formRecords));

         // запуск обработчиков событий на изменения
         for (const id of changedIds) {
            const form = targetFormsMap.get(id)!;
            const oldForm = initialFormsMap.get(id)!;
            handleChangeForm(form, form.formParams, oldForm.formParams, FormChangeType.Edit);
         }
      }

      setUsedTarget(true);
   }, [
      dispatch,
      handleChangeForm,
      initialFormsMap,
      insertForms,
      isEqualForms,
      removeForms,
      targetFormsMap,
      usedTarget,
   ]);

   return {
      isLoadTarget: !(alreadyTarget || usedTarget),
   };
}
