import { IconDefinition } from '@fortawesome/fontawesome-svg-core';
import { isEmpty } from '@yandex-infracloud-ui/libs';
import { FormikErrors, FormikProps, FormikState, validateYupSchema, yupToFormErrors } from 'formik';
import React, { createContext, ReactNode } from 'react';
import { Store } from 'redux';
import { from, Observable } from 'rxjs';
import { map } from 'rxjs/operators';
import { Schema } from 'yup';
import type { RootState } from '../../redux';
import { Entity } from '../../redux/models';
import { pathSorter } from '../../utils';
import { FormStoreRecord, selectFormStates } from '../forms';

/**
 * TODO: @khoden, давай, напишем, что это за файл, что тут происходит (c) @mertas
 */

/**
 * FP - FormParams
 * E - Entity
 * SE - SubEntity
 */
export interface SubForm<FP, E extends Entity, SE extends Entity> {
   /**
    * Timestamp in ms
    */
   changed?: number;
   formParams: FP;
   hasChildren: boolean;
   id: string;
   isAdded?: boolean;
   isRemoved?: boolean;
   levelConfig: FormLevelConfig<FP, E, SE>;
   parentForms: AnySubForm[];
   value: E;
}

/**
 * FP - FormParams
 * E - Entity
 * SE - SubEntity
 */
export interface SubFormProps<FP, E extends Entity, SE extends Entity> {
   disabled: boolean;
   form: SubForm<FP, E, SE>;
   formik: FormikProps<FP>;
   isRootEntityNew: boolean;
   readonly: boolean;
}

/**
 * FP - FormParams
 * E - Entity
 * SE - SubEntity
 * EVP - EmptyValueParams
 */
export interface FormLevelConfig<FP, E extends Entity, SE extends Entity, EVP = any> {
   /**
    * Очистка некоторый полей из модели при клонировании
    *
    * Пока только очищаются delegation_token секретов
    */
   clearOnClone?: (v: E) => void;
   component: React.FC<SubFormProps<FP, E, SE>>;
   getEmptyValue: (params?: EVP) => E;
   formParamsToValue: (params: FP, previousValue: E) => E;
   getChildren: (v: E) => SE[];
   icon: IconDefinition;
   iconChanged: IconDefinition;
   iconColor: string;
   id: string;
   level: number;
   name?: string;
   renderTitle: (v: E) => ReactNode;
   routePath: (id: string) => string;
   validationSchema: Schema<FP>;
   valueToFormParams: (v: E) => FP;
}

export type AnySubForm = SubForm<Entity | any, Entity | any, Entity | any>;

export type AnyFormLevelConfig = FormLevelConfig<Entity | any, Entity | any, Entity | any>;

export enum FormChangeType {
   Edit = 'edit',
   Add = 'add',
   Remove = 'remove',
}

export interface FormChangeParams {
   type: FormChangeType;
   levelId: string;
   path: string;
   newValue: any;
   oldValue: any;
   formValues: any;
   parentForms: AnySubForm[];
}

export interface FormChangeListener {
   listenFields: {
      [levelId: string]: string[]; // value is list of pathes (to form fields) to listen
   };

   onChange(params: FormChangeParams): void;
}

export type AnyLevelParamsMap = Map<number, any>;

export interface HugeFormContextContent {
   disabled: boolean;

   /**
    * Словарь  для быстрого доступа к форме по её id
    */
   formMap: Map<string, AnySubForm>;
   isNew: boolean;
   levelConfigMap: Map<number, AnyFormLevelConfig>;
   /**
    * Словарь для каких-нибудь произвольных значений
    */
   payload: Record<string, any>;
   currentEmptyParams: AnyLevelParamsMap;
   readonly: boolean;
}

export const HugeFormContext = createContext<HugeFormContextContent>({
   disabled: false,
   formMap: new Map(),
   isNew: false,
   levelConfigMap: new Map(),
   payload: {},
   currentEmptyParams: new Map(),
   readonly: false,
});

export interface HugeFormRef<V> {
   getForms(): AnySubForm[];

   getValue(): V;

   resetCurrentForm(): void;
}

export function sortForms(forms: AnySubForm[]) {
   forms.sort((a, b) => pathSorter(a.id.toLowerCase(), b.id.toLowerCase()));
}

export function getChildrenForms(form: AnySubForm, forms: AnySubForm[]): AnySubForm[] {
   return forms.filter(f => f.parentForms[0]?.id === form.id); // Только узлы с таким прямым родителем
}

/**
 * Возвращает соседей формы того же уровня и с тем же родителем
 */
export function getSiblingForms(form: AnySubForm, forms: AnySubForm[]): AnySubForm[] {
   return forms
      .filter(f => f.id !== form.id) // Исключаем её саму
      .filter(f => f.levelConfig.id === form.levelConfig.id) // Только узлы такого же типа
      .filter(f => f.parentForms[0].id === form.parentForms[0].id); // Только узлы с общим родителем
}

export function getCousinForms(form: AnySubForm, forms: AnySubForm[]): AnySubForm[] {
   const parent = form.parentForms[0];
   const grandParent = parent.parentForms[0];
   if (!grandParent) {
      return [];
   }

   const parentSiblings = getSiblingForms(parent, forms);
   const parentSiblingsIds = new Set(parentSiblings.map(s => s.id));

   return forms
      .filter(f => parentSiblingsIds.has(f.parentForms[0]?.id)) // Дети сиблингов родителя
      .filter(f => f.levelConfig.id === form.levelConfig.id); // Того же типа
}

export function getSiblingAndCousinForms(form: AnySubForm, forms: AnySubForm[]): AnySubForm[] {
   return [...getSiblingForms(form, forms), ...getCousinForms(form, forms)];
}

export function getDirectChildForms(parentForm: AnySubForm, forms: AnySubForm[]): AnySubForm[] {
   return forms
      .filter(f => f.levelConfig.level > 0) // У корня нет родителей
      .filter(f => f.parentForms[0].id === parentForm.id);
}

export function getEmptyFormikState<V>(values?: V): FormikState<V> {
   return {
      errors: {},
      isSubmitting: false,
      isValidating: false,
      status: undefined,
      submitCount: 0,
      touched: {},
      values: values ?? ({} as V),
   };
}

export interface ValidationContext {
   form: AnySubForm | undefined;

   childrenForms: AnySubForm[];
   cousinForms: AnySubForm[];
   siblingForms: AnySubForm[];
   parentForms: AnySubForm[];

   // TODO @nodejsgirl
   // кажется, что можно брать списки id прямо на месте из нужных форм?
   siblingIds: Set<string>;
   siblingAndCousinIds: Set<string>;
}

export function getValidationContextForForm(
   form: AnySubForm,
   forms: AnySubForm[],
   formStates: Map<string, FormStoreRecord> | null,
): ValidationContext {
   const getForm = (f: AnySubForm) => (formStates ? formStates.get(f.id)?.state.values ?? f.formParams : f.formParams);
   const getFormId = (f: AnySubForm) =>
      formStates ? formStates.get(f.id)?.state.values.id ?? f.formParams.id : f.formParams.id;

   if (isEmpty(form.parentForms)) {
      return {
         form: getForm(form),

         childrenForms: getChildrenForms(form, forms).map(getForm),
         cousinForms: [],
         siblingForms: [],
         parentForms: [],

         siblingIds: new Set(),
         siblingAndCousinIds: new Set(),
      };
   }

   return {
      form: getForm(form),

      childrenForms: getChildrenForms(form, forms).map(getForm),
      cousinForms: getCousinForms(form, forms).map(getForm),
      siblingForms: getSiblingForms(form, forms).map(getForm),
      parentForms: form.parentForms.map(getForm),

      siblingIds: new Set(getSiblingForms(form, forms).map(getFormId)),
      siblingAndCousinIds: new Set(getSiblingAndCousinForms(form, forms).map(getFormId)),
   };
}

export function getFormsValidationErrors(
   rootFormId: string,
   forms: AnySubForm[],
   reduxStore: Store<RootState> | null,
): Observable<Record<string, FormikErrors<unknown>>> {
   const formStates = reduxStore ? selectFormStates(reduxStore.getState()) : null;

   const getFormValues = ({ id, formParams }: AnySubForm) =>
      formStates ? formStates.get(id)?.state.values ?? formParams : formParams;

   const errorsPromises = Promise.all(
      forms.map(form => {
         const context = getValidationContextForForm(form, forms, formStates);

         return validateYupSchema(getFormValues(form), form.levelConfig.validationSchema, false, context).then(
            () => ({ id: form.id, errors: {} as FormikErrors<unknown> }),
            e => ({ id: form.id, errors: yupToFormErrors(e) }),
         );
      }),
   );

   return from(errorsPromises).pipe(
      map(formErrors =>
         formErrors
            .filter(({ errors }) => !isEmpty(errors))
            .reduce((acc, { id, errors }) => {
               acc[id] = errors;

               return acc;
            }, {} as Record<string, FormikErrors<unknown>>),
      ),
   );
}
