import { FormContext, FormHooks, FormHooksContext, Loader } from '@yandex-infracloud-ui/libs';
import { Form, Formik, FormikProps } from 'formik';
import { FormikHelpers } from 'formik/dist/types';
import React, { MutableRefObject, ReactNode, useMemo } from 'react';
import { Schema } from 'yup';

import classes from './FormPage.module.css';
import { FormPageLayout } from './FormPageLayout';

interface Props<T> {
   autoComplete?: 'off';
   cancelButtonText?: string;
   formContext?: any;
   formRef?: MutableRefObject<FormikProps<T> | undefined>;
   initialValues: T;
   loading?: string | null;
   subTitle: ReactNode;
   submitButtonText?: string;
   title: string;
   validationSchema?: Schema<T>;
   customLayout?: boolean;
   hideContentPaddings?: boolean;

   onFieldChange?<K extends keyof T>(field: K, value: T[K]): void;

   onSubmit(values: T, formikHelpers: FormikHelpers<T>): void | Promise<any>;

   onCancel(): void;

   children(form: FormikProps<T>): ReactNode;
}

/**
 * Полностраничная форма с кнопками в шапке.
 *
 * Из-за того, что кнопка должна быть внутри формы, чтобы сработал нативный submit, сама форма тоже здесь заводится
 */
export const FormPage: React.FC<Props<any>> = React.memo(
   ({
      subTitle,
      title,
      cancelButtonText = 'Cancel',
      submitButtonText = 'Save',
      autoComplete,
      loading,
      customLayout = false,
      hideContentPaddings = false,

      formRef,
      initialValues,
      validationSchema,
      formContext,
      onSubmit,
      onFieldChange,
      onCancel,
      children,
   }) => {
      const formHooks = useMemo(() => ({ onFieldChange } as FormHooks<any>), [onFieldChange]);

      if (loading) {
         // initialValues может быть еще не загружен, поэтому инициализировать форму рано
         return (
            <FormPageLayout
               cancelButtonText={cancelButtonText}
               disabled={true}
               hideContentPaddings={hideContentPaddings}
               onCancel={onCancel}
               submitButtonText={submitButtonText}
               supTitle={subTitle}
               title={title}
            >
               <Loader text={loading} cls={classes.loader} align={'left'} />
            </FormPageLayout>
         );
      }

      return (
         <Formik initialValues={initialValues} onSubmit={onSubmit} validationSchema={validationSchema}>
            {form => {
               if (formRef) {
                  formRef.current = form;
               }

               return (
                  <FormHooksContext.Provider value={formHooks}>
                     <FormContext.Provider value={formContext}>
                        <Form autoComplete={autoComplete}>
                           {customLayout ? (
                              children(form)
                           ) : (
                              <FormPageLayout
                                 cancelButtonText={cancelButtonText}
                                 disabled={form.isSubmitting}
                                 hideContentPaddings={hideContentPaddings}
                                 onCancel={onCancel}
                                 submitButtonText={submitButtonText}
                                 supTitle={subTitle}
                                 title={title}
                              >
                                 {children(form)}
                              </FormPageLayout>
                           )}
                        </Form>
                     </FormContext.Provider>
                  </FormHooksContext.Provider>
               );
            }}
         </Formik>
      );
   },
);

FormPage.displayName = 'FormPage';
