import React, { Fragment, ReactNode, useCallback, useMemo, useState } from 'react';

import { Checkbox, ControlGroupOption } from '@yandex-cloud/uikit';
import { EnumSwitcher } from '@yandex-infracloud-ui/libs';
import { Diff, DiffViewType, Hunk, HunkModel, parseDiff, tokenize } from 'react-diff-view';
import refractor from 'refractor/core';
import yamlSyntax from 'refractor/lang/yaml';

import { diffLines, formatLines } from 'unidiff';

import classes from './DiffView.module.css';

refractor.register(yamlSyntax);

interface Props {
   fullContext?: boolean;
   language: string;
   leftToolbar?: ReactNode;
   modified: string;
   original: string;
   rightToolbar?: ReactNode;
   unified?: boolean;
}

const viewTypeOptions: ControlGroupOption[] = [
   { content: 'unified', value: 'unified' },
   { content: 'split', value: 'split' },
];

export const DiffView: React.FC<Props> = ({
   language,
   leftToolbar,
   modified,
   original,
   rightToolbar,
   unified: initialUnified = true,
   fullContext: initialFullContext = false,
}) => {
   const [fullContext, setFullContext] = useState(initialFullContext);
   const [viewType, setViewType] = useState<DiffViewType>(initialUnified ? 'unified' : 'split');

   const handleViewTypeChange = useCallback((v: string) => {
      setViewType(v as DiffViewType);
   }, []);

   const handleFullContextChange = useCallback(() => {
      setFullContext(v => !v);
   }, []);

   const { tokens, diff } = useMemo(() => {
      const context = fullContext ? Math.max(original.length, modified.length) : 5;
      const diffText = formatLines(diffLines(original, modified), { context });
      const firstDiff = parseDiff(diffText, { nearbySequences: 'zip' })[0];

      return {
         tokens: tokenize(firstDiff.hunks, {
            highlight: true,
            refractor,
            oldSource: original,
            language,
            enhancers: [],
         }),
         diff: firstDiff,
      };
   }, [fullContext, language, modified, original]);

   return (
      <div className={classes.wrapper}>
         <div className={classes.toolbar} data-test={'DiffView:Toolbar'}>
            <div className={classes.leftToolbar}>{leftToolbar}</div>

            <div className={classes.rightToolbar}>
               {rightToolbar && <div>{rightToolbar}</div>}

               <div data-test={'DiffView:Type'}>
                  <EnumSwitcher
                     name={'viewType'}
                     value={viewType}
                     onChange={handleViewTypeChange}
                     options={viewTypeOptions}
                  />
               </div>

               <div data-test={'DiffView:FullContext'}>
                  <Checkbox checked={fullContext} onChange={handleFullContextChange}>
                     Full context
                  </Checkbox>
               </div>
            </div>
         </div>

         <div className={classes.diffContent}>
            <Diff viewType={viewType} diffType={diff.type} hunks={diff.hunks} tokens={tokens} gutterType={'default'}>
               {(hunks: HunkModel[]) =>
                  hunks.map((hunk, i) => (
                     <Fragment key={hunk.content}>
                        <Hunk hunk={hunk} />
                        {i < hunks.length - 1 ? (
                           <tbody>
                              <tr>
                                 <td colSpan={viewType === 'split' ? 4 : 3}>
                                    <div className={classes.separator} />
                                 </td>
                              </tr>
                           </tbody>
                        ) : null}
                     </Fragment>
                  ))
               }
            </Diff>
         </div>
      </div>
   );
};

DiffView.displayName = 'DiffView';
