import { CheckboxSelection, classNames } from '@yandex-infracloud-ui/libs';
import React, { useMemo, useState } from 'react';

import { LegoCheckBox } from '../../../_lego';

import { convertToRows, isColumnHidden } from './convertToRows';
import classes from './HierarchicalTable.module.css';
import { Row } from './models';
import { HierarchicalTableConfig } from './publicModels';

interface Props {
   className?: string;
   config: HierarchicalTableConfig;
   context: any;
   disabled: boolean;
   readonly: boolean;
   values: any[];
   'data-e2e': string;
}

export const HierarchicalTable: React.FC<Props> = React.memo(
   ({ className, values, config, context, disabled, readonly, 'data-e2e': dataE2E }) => {
      const firstLevelConfig = config.levels[0];
      const allIds = useMemo(() => values.map(firstLevelConfig.getId), [firstLevelConfig.getId, values]);

      const [selectedIds, setSelectedIds] = useState<Set<string | number>>(new Set());

      const rows: Row[] = useMemo(
         () =>
            convertToRows({
               addCheckbox: config.addCheckbox && !readonly,
               context,
               disabled,
               levelConfigs: config.levels,
               parent: null,
               values,
            }),
         [config.addCheckbox, config.levels, context, disabled, readonly, values],
      );

      return (
         <CheckboxSelection allIds={allIds} value={selectedIds} onChange={setSelectedIds}>
            {({ isAllSelected, toggleAll }) => (
               <table className={classNames(classes.table, className)} data-e2e={dataE2E}>
                  <thead>
                     <tr>
                        {config.addCheckbox && !readonly ? (
                           <th className={classes.checkbox} key={'checkbox'}>
                              <LegoCheckBox
                                 checked={isAllSelected !== false}
                                 indeterminate={isAllSelected === null}
                                 onChange={toggleAll}
                                 disabled={disabled}
                              />
                           </th>
                        ) : null}

                        {config.levels.map(level => (
                           <React.Fragment key={level.id}>
                              {level.columns
                                 .filter(column => !isColumnHidden(column, context))
                                 .map(column => (
                                    <th key={column.id} className={column.className}>
                                       {column.head}
                                    </th>
                                 ))}
                           </React.Fragment>
                        ))}
                     </tr>
                  </thead>

                  <tbody>
                     {rows.map(row => (
                        <tr key={row.id} className={row.className}>
                           {row.cells.map(cell => (
                              <td
                                 key={cell.id}
                                 rowSpan={cell.rowSpan}
                                 colSpan={cell.colSpan}
                                 className={cell.className}
                              >
                                 {cell.content}
                              </td>
                           ))}
                        </tr>
                     ))}
                  </tbody>
               </table>
            )}
         </CheckboxSelection>
      );
   },
);

HierarchicalTable.displayName = 'HierarchicalTable';
