import { convertToRows } from './convertToRows';
import { HierarchicalTableConfig, LevelConfig } from './publicModels';

describe('components/lib/HierarchicalTable/convertToRows', () => {
   it('should works', () => {
      expect(2 + 2).toBe(4);
   });

   it('should works for one level', () => {
      interface Value {
         id: string;
         name: string;
      }

      const values = [
         { id: 'sss', name: 'xxx' },
         { id: 'sss2', name: 'xxx' },
         { id: 'sss3', name: 'xxx' },
      ];

      const config: HierarchicalTableConfig = {
         addCheckbox: true,
         levels: [
            {
               id: 'name',
               columns: [
                  {
                     id: 'name',
                     head: 'Name',
                     render: v => v.name,
                  },
               ],
               getId: v => v.id,
            } as LevelConfig<void, Value, void, any>,
         ],
      };

      const result = convertToRows({
         addCheckbox: true,
         context: null,
         disabled: false,
         levelConfigs: config.levels,
         parent: undefined,
         values,
      });

      expect(result.length).toBe(3);
      expect(result[0].id).toBe('sss');
      expect(result[0].cells.length).toBe(2);
      expect(result[1].id).toBe('sss2');
      expect(result[2].id).toBe('sss3');
   });

   it('should works for one level and several columns', () => {
      interface Value {
         age: number;
         id: string;
         name: string;
      }

      const values = [
         { id: 'sss', name: 'xxx', age: 1 },
         { id: 'sss2', name: 'yyy', age: 2 },
         { id: 'sss3', name: 'zzz', age: 3 },
      ];

      const config: HierarchicalTableConfig = {
         addCheckbox: true,
         levels: [
            {
               id: 'person',
               columns: [
                  {
                     id: 'name',
                     head: 'Name',
                     render: v => v.name,
                  },
                  {
                     id: 'age',
                     head: 'Age',
                     render: v => v.age,
                  },
               ],
               getId: v => v.id,
            } as LevelConfig<void, Value, void, any>,
         ],
      };

      const result = convertToRows({
         addCheckbox: true,
         context: null,
         disabled: false,
         levelConfigs: config.levels,
         parent: undefined,
         values,
      });

      expect(result.length).toBe(3);
      expect(result[0].cells.length).toBe(3);
      expect(result[0].cells[1].content).toBe('xxx');
      expect(result[0].cells[2].content).toBe(1);
   });

   it('should works for two levels', () => {
      interface Item {
         itemId: number;
         name: string;
      }

      interface Value {
         id: string;
         items: Item[];
         name: string;
      }

      const values: Value[] = [
         {
            id: 'sss',
            name: 'xxx',
            items: [
               { itemId: 23, name: 'Child1' },
               { itemId: 21, name: 'Child2' },
            ],
         },
         { id: 'sss2', name: 'yyy', items: [{ itemId: 23, name: 'Child1' }] },
         { id: 'sss3', name: 'zzz', items: [{ itemId: 23, name: 'Child1' }] },
      ];

      const config: HierarchicalTableConfig = {
         addCheckbox: true,
         levels: [
            {
               id: 'name',
               columns: [
                  {
                     id: 'name',
                     head: 'Name',
                     render: v => v.name,
                  },
               ],
               getChildren: v => v.items,
               getId: v => v.id,
            } as LevelConfig<void, Value, Item, any>,
            {
               id: 'child',
               getId: v => v.itemId,
               columns: [
                  {
                     id: 'child',
                     head: 'Child',
                     render: v => v.name,
                  },
               ],
            } as LevelConfig<Value, Item, void, any>,
         ],
      };

      const result = convertToRows({
         addCheckbox: true,
         context: null,
         disabled: false,
         levelConfigs: config.levels,
         parent: undefined,
         values,
      });

      expect(result.length).toBe(4);
      expect(result[0].cells.length).toBe(3);
      expect(result[0].cells[1].content).toBe('xxx');
      expect(result[0].cells[1].rowSpan).toBe(2);
      expect(result[0].cells[2].content).toBe('Child1');
   });
});
