import React, { ReactNode, useCallback, useEffect, useRef } from 'react';

import { classNames, modalService, useDismounted } from '@yandex-infracloud-ui/libs';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faEye } from '@fortawesome/pro-regular-svg-icons';
import { takeUntil } from 'rxjs/operators';

import { SimpleModal } from '../SimpleModal/SimpleModal';

import classes from './MiniBox.module.css';
import { useForceRender } from '../../../utils';

export interface MiniBoxProps {
   /**
    * ширина видимой части для css, в пикселях либо строкой css
    */
   width?: number | string;
   /**
    * высота видимой части для css, в пикселях либо строкой css
    */
   height?: number | string;

   /**
    * подпись для кнопки раскрытия
    */
   showMoreTitle?: ReactNode;

   /**
    * подпись для показа текста целиком
    */
   fullTextTitle?: string;

   preferenceMode?: 'text' | 'block';
}

export const MiniBox: React.FC<MiniBoxProps> = ({
   children,
   width = '200px',
   height = '1.5rem',
   showMoreTitle = <FontAwesomeIcon icon={faEye} />,
   fullTextTitle = '',
   preferenceMode = 'text',
}) => {
   const dismounted = useDismounted();

   const onMoreClick = useCallback(() => {
      modalService
         .open(
            SimpleModal,
            { title: fullTextTitle, children: <div className={classes.modalContent}>{children}</div> },
            {
               dialogProps: { size: 'm' },
            },
         )
         .pipe(takeUntil(dismounted))
         .subscribe();
   }, [children, dismounted, fullTextTitle]);

   const cssHeight = typeof height === 'number' ? `${height}px` : height;
   const cssWidth = typeof width === 'number' ? `${width}px` : width;

   // DOM контента
   const contentRef = useRef<null | HTMLDivElement>(null);

   // DOM блока-измерителя
   const sizeRef = useRef<null | HTMLDivElement>(null);

   // контент не влезает
   const isContentBig = useRef(false);

   const forceRender = useForceRender();
   useEffect(() => {
      // активация первого пересчёта isContentBig
      forceRender();
   }, [forceRender]);

   useEffect(() => {
      const { offsetHeight: contentHeight = 0, offsetWidth: contentWidth = 0 } = contentRef.current ?? {};
      const { offsetHeight: sizeHeight = 0, offsetWidth: sizeWidth = 0 } = sizeRef.current ?? {};
      // контент не влезает в блок, если не влезает любая из сторон
      isContentBig.current = contentHeight > sizeHeight || contentWidth > sizeWidth;
   });

   return (
      <div
         className={classNames(classes.container, preferenceMode === 'text' ? classes.textContainer : undefined)}
         style={{ maxHeight: cssHeight, maxWidth: cssWidth }}
      >
         {/* блок-измеритель, с помощью него меряем абсолютные размеры в пискелях */}
         <div
            ref={sizeRef}
            className={classes.sizeBlock}
            style={{
               height: '100%',
               width: '100%',
            }}
         >
            {' '}
         </div>
         {/* блок с контентом, на который и накладываются ограничения по размерам */}
         <div
            className={preferenceMode === 'text' ? classes.content : undefined}
            style={{ maxHeight: '100%', maxWidth: '100%' }}
         >
            {/* блок, который растягивается по содержимому, используется для подсчёта размеров контента */}
            <div ref={contentRef} style={{ display: preferenceMode === 'text' ? 'inline' : 'block' }}>
               {children}
            </div>
         </div>
         {/* если контент не влезает, рисуем кнопку для показа целиком */}
         {isContentBig.current && (
            <button
               className={classNames(classes.more, preferenceMode === 'block' ? classes.accentMore : undefined)}
               onClick={event => {
                  event.stopPropagation();
                  onMoreClick();
               }}
               type={'button'}
               title={'show more'}
               style={{ maxHeight: '100%', maxWidth: '100%' }}
            >
               {showMoreTitle}
            </button>
         )}
      </div>
   );
};
