import { classNames } from '@yandex-infracloud-ui/libs';
import React, { ReactNode } from 'react';

import classes from './Status.module.css';

export enum StatusType {
   Primary = 'primary',
   Secondary = 'secondary',
}

export enum StatusState {
   Ok = 'ok',
   Error = 'error',
   Progress = 'progress',
   Inactive = 'inactive',
   Planned = 'planned',
   Pause = 'pause',
   Default = 'default',
   Unknown = 'unknown',
}

interface Props {
   state: StatusState;
   type?: StatusType;
   isAnimated?: boolean;
   className?: string;
   sign?: boolean | ReactNode;
}

function renderSign(sign: boolean | ReactNode, state: StatusState): ReactNode {
   if (typeof sign === 'boolean') {
      return {
         [StatusState.Ok]: 'v',
         [StatusState.Error]: '!',
         [StatusState.Progress]: '...',
         [StatusState.Inactive]: 'x',
         [StatusState.Planned]: '~',
         [StatusState.Default]: '-',
         [StatusState.Unknown]: '?',
         [StatusState.Pause]: '||',
      }[state];
   }
   return sign ?? '-';
}

export const stateStyles: Record<StatusState, string> = {
   [StatusState.Ok]: classes.ok,
   [StatusState.Error]: classes.error,
   [StatusState.Progress]: classes.progress,
   [StatusState.Inactive]: classes.inactive,
   [StatusState.Planned]: classes.planned,
   [StatusState.Default]: classes.default,
   [StatusState.Unknown]: classes.unknown,
   [StatusState.Pause]: classes.pause,
};

export const Status: React.FC<Props> = React.memo(
   ({ type = 'secondary', state, isAnimated = true, sign, className, children }) => {
      let content: ReactNode;

      if (type === StatusType.Primary) {
         content = (
            <div
               className={classNames(classes.primary, classes.content, {
                  [classes.primaryAnimated]: state === StatusState.Progress,
               })}
            >
               {children ?? state}
            </div>
         );
      } else if (sign) {
         content = (
            <div className={classes.secondary}>
               <div className={classes.sign}>{renderSign(sign, state)}</div>
               <div className={classes.content}>{children ?? state}</div>
            </div>
         );
      } else {
         const circle = <div className={classes.circle} />;
         if (isAnimated && state === StatusState.Progress) {
            content = (
               <div className={classes.secondary}>
                  <div className={classes.circlePulseContainer}>
                     {circle}
                     <div className={classNames(classes.circlePulse, classes.animated)} />
                  </div>
                  <div className={classes.content}>{children ?? state}</div>
               </div>
            );
         } else {
            content = (
               <div className={classes.secondary}>
                  {circle}
                  <div className={classes.content}>{children ?? state}</div>
               </div>
            );
         }
      }

      return <div className={classNames(classes.status, stateStyles[state], className)}>{content}</div>;
   },
);

Status.displayName = 'Status';
