import React, { ReactNode, useCallback } from 'react';

import { faAngleUp } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Card } from '@yandex-cloud/uikit';

import classes from './TilesTabs.module.scss';

const TILE_GAP = 8;
const TILE_PADDING = 12;
const TILE_CARD_WIDTH = 600;

export interface TilesTab {
   tile: ReactNode;
   content: ReactNode;
}

interface Props {
   items: TilesTab[];
   selectedTab: number | null;
   onToogle(index: number | null): void;
   tileMinWidth?: number;
   tileContentWidth?: number;
   inline?: boolean;
   showCollapse?: boolean;
}

export const TilesTabs: React.FC<Props> = React.memo(
   ({
      items,
      tileMinWidth = 300,
      tileContentWidth = TILE_CARD_WIDTH,
      inline = false,
      showCollapse = true,
      selectedTab,
      onToogle,
   }: Props) => {
      const onTabClick = useCallback(
         (index: number) => {
            onToogle(index);
         },
         [onToogle],
      );

      const onCollapseClick = useCallback(() => {
         onToogle(null);
      }, [onToogle]);

      const size = items.length;

      const tilePadding = inline ? 0 : TILE_PADDING;

      // ширина, как если бы все плитки были в один ряд
      // чтобы компонент не был шире родительского, максимальная ширина ограничена
      const tilesWidth = tileMinWidth * size + (size > 1 ? TILE_GAP * (size - 1) : 0) + tilePadding * 2;
      const width = Math.max(tilesWidth, selectedTab !== null ? tileContentWidth + TILE_GAP + 2 * tilePadding : 0);

      return (
         <div style={{ minWidth: `${width}px`, maxWidth: '100%' }}>
            <Card view={inline ? 'clear' : 'raised'} className={classes.tabsArea}>
               <div
                  className={classes.tabs}
                  style={{
                     gridTemplateColumns: `repeat(auto-fill, minmax(${tileMinWidth}px, 1fr))`,
                     gap: `${TILE_GAP}px`,
                     padding: `${tilePadding}px`,
                  }}
               >
                  {items.map((item, i) => (
                     <Card
                        // eslint-disable-next-line react/no-array-index-key
                        key={i}
                        type={'selection'}
                        view={'outlined'}
                        selected={selectedTab === i}
                        className={classes.tab}
                        onClick={() => onTabClick(i)}
                     >
                        <div role={'button'} tabIndex={0} style={{ minWidth: `${tileMinWidth}px` }}>
                           {item.tile}
                        </div>
                     </Card>
                  ))}
               </div>
               {selectedTab !== null && (
                  <>
                     {items[selectedTab] && <div className={classes.content}>{items[selectedTab].content}</div>}
                     {showCollapse && (
                        <button className={classes.collapse} onClick={onCollapseClick} type={'button'}>
                           <span className={classes.collapseIcon}>
                              <FontAwesomeIcon icon={faAngleUp} />
                           </span>
                           Collapse
                        </button>
                     )}
                  </>
               )}
            </Card>
         </div>
      );
   },
);

TilesTabs.displayName = 'TilesTabs';
