import { fromQuery, useDismounted } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { ReactNode, useCallback, useEffect, useMemo, useRef } from 'react';
import { useLocation } from 'react-router';
import { map, Observable, takeUntil } from 'rxjs';

import {
   columnPresets,
   DEFAULT_LIMIT,
   DEFAULT_ORDER,
   GetItemsResponse,
   LimitFilter,
   PlainObject,
   SmartTable,
} from '../smart-table';
import { getTableRowId } from '../smart-table/components/Table/Table';
import { OrderFilter } from '../smart-table/filters/OrderFilter/OrderFilter';
import { SmartTableRef } from '../smart-table/SmartTable';

import { LogApi } from './api/LogApi';
import { currentTokenSymbol, LogEntry } from './api/models/LogEntry';
import { intervalToRange, RequestParams } from './api/models/RequestParams';
import { SearchLogEntriesRequest } from './api/models/SearchLogEntriesRequest';
import classes from './BaseLogPage.module.css';
import { TimeCell } from './cells/TimeCell/TimeCell';
import { ClickableJson } from './components/ClickableJson/ClickableJson';
import { IntervalFilter, intervalFilterName } from './filters/IntervalFilter/IntervalFilter';
import { QueryFilter } from './filters/QueryFilter/QueryFilter';
import { Operator, queryOperatorMap, QueryValue } from './query';
import { getCleanExpressions } from './query/provider/helpers';
import { QueryField } from './query/provider/models';
import { QueryProvider } from './query/provider/QueryProvider';
import { LogPageContextValue, LogPageProvider } from './useLogPageContext';

// пустые строки и строки, содержащие пробелы и спецсимволы, должны быть в кавычках
const quoteValueIfNeed = (value: any) =>
   typeof value === 'string' && (value === '' || /["\s,!=;]/.test(value)) ? `"${value.replace(/"/g, '\\"')}"` : value;

const ST = SmartTable;
const cp = columnPresets;

interface Props {
   apiEndpoint: string;
   columns: ReactNode;
   filters?: ReactNode;
   forcedFilters?: RequestParams;
   requiredFilters?: string[];
   idColumns?: string[];
   queryString: string;
   hideColumnPresets?: boolean; // DEPLOY-5824
   filtersClassName?: string;
   requestFields?: QueryField[];
   customRequestFields?: QueryField[];
   snakeCase?: boolean;

   onQueryStringUpdate(qs: string): void;

   extraSecondRow?(item: any): void;
}

export const BaseLogPage: React.FC<Props> = React.memo(
   ({
      apiEndpoint,
      columns,
      extraSecondRow,
      filters: filterNodes,
      forcedFilters,
      requiredFilters = [],
      idColumns = ['timestamp'],
      onQueryStringUpdate,
      queryString,
      // hideColumnPresets, // DEPLOY-5824
      filtersClassName,
      requestFields = [],
      customRequestFields = [],
      snakeCase = false,
   }) => {
      const smartTableRef = useRef<SmartTableRef>(null);
      const dismounted = useDismounted();

      const api = useMemo(() => new LogApi(apiEndpoint), [apiEndpoint]);
      const queryProvider = useMemo(() => new QueryProvider(api, forcedFilters, requestFields, customRequestFields), [
         api,
         forcedFilters,
         requestFields,
         customRequestFields,
      ]);

      const filtersToRequestParams = useCallback(
         (filters: PlainObject) => {
            const { interval, query, limit, ...restFilters } = filters;

            const timestampRange = intervalToRange(interval);

            const expressions = getCleanExpressions(query.query);

            const requestParamsFromQuery = queryProvider.buildRequestParams({ addForcedParams: false, expressions });

            const requestParamsFromFilters = queryProvider.buildRequestParamsFromFilters({
               addForcedParams: false,
               filters: restFilters,
            });

            return {
               limit,
               timestampRange,
               ...requestParamsFromQuery,
               ...requestParamsFromFilters,
            };
         },
         [queryProvider],
      );

      const { search } = useLocation();
      const { range, highlight } = fromQuery(search);

      const getDefaultOpenedRowIdList = useCallback(
         (items: LogEntry[]) => {
            if (highlight && range) {
               return items
                  .filter((item, i) => i.toString() === highlight && item[currentTokenSymbol] === range)
                  .map(v => getTableRowId(v, idColumns));
            }

            return [];
         },
         [highlight, range, idColumns],
      );

      const getItems = useCallback(
         (filters: PlainObject, nextToken?: string): Observable<GetItemsResponse> => {
            const params: SearchLogEntriesRequest = {
               continuationToken: nextToken ?? (range as string), // range - токен из ссылки
               ...filtersToRequestParams(filters),
               ...forcedFilters,
            };

            if (snakeCase) {
               return api.searchLogEntriesSnake(params).pipe(
                  map(
                     resp =>
                        ({
                           nextToken: resp.continuation_tokens?.forward,
                           items: resp.log_entries,
                        } as GetItemsResponse),
                  ),
                  takeUntil(dismounted),
               );
            }

            return api.searchLogEntries(params).pipe(
               map(
                  resp =>
                     ({
                        nextToken: resp.continuationTokens?.forward,
                        items: resp.logEntries,
                     } as GetItemsResponse),
               ),
               takeUntil(dismounted),
            );
         },
         [api, dismounted, filtersToRequestParams, forcedFilters, range, snakeCase],
      );

      const handleFieldClick = useCallback((field: string, operator: Operator, value: any) => {
         // TODO smarter, not just adding to the end #DEPLOY-5956
         const newFilterExpression = `${field}${queryOperatorMap[operator]}${quoteValueIfNeed(value)}; `;

         const hooks = smartTableRef.current?.getFiltersHook();
         if (hooks) {
            const query = hooks.getFilter('query') as QueryValue;
            const newQuery = [query.query, newFilterExpression].filter(Boolean).join(' ');
            hooks.setFilter('query', { ...query, query: newQuery });
         } else {
            console.error("The filters' hooks MUST be in context", newFilterExpression);
         }
      }, []);

      const renderSecondRow = useCallback(
         (item: any) => (
            <>
               <section>
                  <b>RAW:</b>
                  <ClickableJson value={item} onSelect={handleFieldClick} />
               </section>

               {extraSecondRow?.(item)}
            </>
         ),
         [extraSecondRow, handleFieldClick],
      );

      const getCurrentFilters = smartTableRef.current?.getFiltersHook()?.getCurrentFilters;

      useEffect(() => {
         queryProvider.init();
      }, [queryProvider]);

      // Обновление текущих фильтров в queryProvider
      useEffect(() => {
         if (smartTableRef.current && getCurrentFilters) {
            queryProvider.setCurrentRequestParams(filtersToRequestParams(getCurrentFilters()));
         }
      }, [filtersToRequestParams, getCurrentFilters, queryProvider]);

      const logPageContext = useMemo(() => ({ queryProvider } as LogPageContextValue), [queryProvider]);

      const handleKeyDown = useCallback((e: React.KeyboardEvent) => {
         if (e.code === 'Enter' && e.ctrlKey) {
            if (smartTableRef.current) {
               const { submitFilters } = smartTableRef.current.getFiltersHook();
               submitFilters();
            }
         }
      }, []);

      const defaultQueryValue: QueryValue = useMemo(() => ({ query: '', request: {} }), []);

      return (
         <LogPageProvider value={logPageContext}>
            <SmartTable
               queryString={queryString}
               requiredFilters={requiredFilters}
               onQueryStringUpdate={onQueryStringUpdate}
               getItems={getItems}
               ref={smartTableRef}
            >
               <ST.Filters className={filtersClassName}>
                  <div className={classes.row}>
                     {filterNodes}

                     <ST.Filter
                        name={intervalFilterName}
                        defaultValue={{}}
                        component={IntervalFilter}
                        getValueFromUrlParams={IntervalFilter.getValueFromUrlParams}
                        setUrlParamsFromValue={IntervalFilter.setUrlParamsFromValue}
                     />

                     {/* TODO: починить или выпилить DEPLOY-5824 */}
                     {/* {!hideColumnPresets ? <ST.ColumnPresets /> : null} */}

                     <ST.Filter name={'limit'} defaultValue={DEFAULT_LIMIT} component={LimitFilter} />

                     <ST.Filter name={'order'} defaultValue={DEFAULT_ORDER} component={OrderFilter} />

                     <div className={classes.buttons}>
                        <ST.FilterButtons />
                     </div>

                     <div className={classes.splitter} />
                  </div>

                  {/* eslint-disable-next-line jsx-a11y/no-static-element-interactions */}
                  <div className={classes.row} onKeyDown={handleKeyDown}>
                     <ST.Filter
                        name={'query'}
                        defaultValue={defaultQueryValue}
                        component={QueryFilter}
                        getValueFromUrlParams={QueryFilter.getValueFromUrlParams}
                        setUrlParamsFromValue={QueryFilter.setUrlParamsFromValue}
                     />
                  </div>
               </ST.Filters>

               <ST.Table
                  idColumns={idColumns}
                  getDefaultOpenedRowIdList={getDefaultOpenedRowIdList}
                  orderColumn={'timestamp'}
                  renderSecondRow={renderSecondRow}
               >
                  <ST.Column name={'timestamp'} header={'Time'} component={TimeCell} presets={cp.default} />
                  {columns}
               </ST.Table>
            </SmartTable>
         </LogPageProvider>
      );
   },
);

BaseLogPage.displayName = 'BaseLogPage';
