import { classNames, isEmpty, sortHandler } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useEffect, useMemo, useState } from 'react';

import { safeParseJson } from '../../helpers/safeParseJson';
import { Operator } from '../../query';
import { useLogPageContext } from '../../useLogPageContext';

import { ClickableJsonValue } from './__value/ClickableJsonValue';

import classes from './ClickableJson.module.css';

interface FieldValueProps {
   path: string;
   value: string | Record<string, any>;
   clickable: boolean;
   greppable: boolean;

   onSelect(field: string, operator: Operator, value: any): void;
}

export interface ClickableJsonProps {
   className?: string;
   value: Record<string, any>;

   onSelect(field: string, operator: Operator, value: any): void;
}

const JsonValue: React.FC<FieldValueProps> = React.memo(({ path, value, clickable, greppable, onSelect }) => {
   // для читабельности и кликабельности json значений (deploy message/context) DEPLOY-5960
   const formattedValue = useMemo(
      () => (typeof value === 'object' ? value : (safeParseJson(value) as string | Record<string, any>)),
      [value],
   );

   if (typeof formattedValue === 'object') {
      const entries = Object.entries(formattedValue);

      return (
         <>
            <span className={classes.operator}>{'{'}</span>
            {entries.map(([key, v], i) => (
               <div className={classes.field} key={`${path}.${key}`}>
                  <span className={classes.key}>{key}</span>
                  <span className={classes.operator}>:</span>{' '}
                  <JsonValue
                     path={`${path}.${key.replaceAll('.', '\\.')}`} // replace . => \.
                     value={v}
                     clickable={clickable}
                     greppable={greppable}
                     onSelect={onSelect}
                  />
                  {entries.length - 1 === i ? '' : <span className={classes.operator}>,</span>}
               </div>
            ))}
            <span className={classes.operator}>{'}'}</span>
         </>
      );
   }

   return (
      <ClickableJsonValue
         className={classes.value}
         clickable={clickable}
         greppable={greppable}
         onClick={o => onSelect(path, o, value)}
      >
         {value}
      </ClickableJsonValue>
   );
});

export const ClickableJson: React.FC<ClickableJsonProps> = React.memo(({ className, value, onSelect }) => {
   const entries = Object.entries(value);
   const [clickableFields, setClickableFields] = useState(new Set<string>());
   const [grepFields, setGrepFields] = useState(new Set<string>());

   const { queryProvider } = useLogPageContext()!;

   useEffect(() => {
      queryProvider.init(isSuccess => {
         if (isSuccess) {
            const { fields } = queryProvider;

            setClickableFields(new Set(fields.filter(f => !isEmpty(f.operators)).map(f => f.name)));
            setGrepFields(new Set(fields.filter(f => f.operators.includes(Operator.Grep)).map(f => f.name)));
         }
      });
   }, [queryProvider]);

   return (
      <div className={classNames(classes.wrapper, className)}>
         <div className={classes.operator}>{'{'}</div>
         {entries
            .sort(([key1], [key2]) => sortHandler(key1, key2))
            .map(([key, v], i) => (
               <div className={classes.field} key={key}>
                  <span className={classes.key}>{key}</span>
                  <span className={classes.operator}>:</span>{' '}
                  <JsonValue
                     path={key}
                     value={v}
                     clickable={clickableFields.has(key) || key === 'context'} // deploy context clickable values DEPLOY-5960
                     greppable={grepFields.has(key)}
                     onSelect={onSelect}
                  />
                  {entries.length - 1 === i ? '' : <span className={classes.operator}>,</span>}
               </div>
            ))}
         <div className={classes.operator}>{'}'}</div>
      </div>
   );
});

ClickableJson.displayName = 'ClickableJson';
