import { RangeDatepickerOutput, SimpleDatepickerOutput, ThinTimelineControls } from '@yandex-data-ui/common';
import { dateTimeParse } from '@yandex-data-ui/date-utils';
import * as React from 'react';
import { useCallback, useEffect, useState } from 'react';

import { FilterProps, PlainObject } from '../../../smart-table/models';

import classes from './IntervalFilter.module.css';

export const intervalFilterName = 'interval';

const DEFAULT_REFRESH_INTERVAL = 0; // 15 * TIMES_IN_MS.Second;

const DEFAULT_RELATIVE_FROM = `now-3h`;
const DEFAULT_RELATIVE_TO = 'now';

export const getDateUrlFormat = (d: Date) => (d ? Math.round(d.getTime() / 1000) : undefined);

function parseDatepickerValue(d: SimpleDatepickerOutput): Date | undefined {
   return dateTimeParse(d.date!, { timeZone: d.timeZone })?.toDate();
}

export function getAbsoluteDateFromRelative(relative: string): Date | undefined {
   return parseDatepickerValue({ type: 'relative', date: relative });
}

interface IntervalItemValue {
   absolute?: Date; // API, links
   relative?: string; // DatePicker
}

export interface IntervalValue {
   from?: IntervalItemValue;
   to?: IntervalItemValue;
}

const getDatePickerValue = (value: IntervalValue): RangeDatepickerOutput => ({
   from:
      value.from?.absolute && !value.from?.relative
         ? { type: 'absolute', date: value.from.absolute.toISOString() }
         : { type: 'relative', date: value.from?.relative ?? DEFAULT_RELATIVE_FROM },
   to:
      value.to?.absolute && !value.to?.relative
         ? { type: 'absolute', date: value.to.absolute.toISOString() }
         : { type: 'relative', date: value.to?.relative ?? DEFAULT_RELATIVE_TO },
});

const IntervalFilterImpl: React.FC<FilterProps<IntervalValue>> = ({ value, onUpdate }) => {
   const [refreshInterval, setRefreshInterval] = useState(DEFAULT_REFRESH_INTERVAL);

   const [{ from, to }, onPickerUpdate] = useState<RangeDatepickerOutput>(getDatePickerValue(value));

   useEffect(() => {
      // empty/relative values, reset filters
      if (!value.from?.absolute && !value.to?.absolute) {
         const relative = {
            from: value.from?.relative ?? DEFAULT_RELATIVE_FROM,
            to: value.to?.relative ?? DEFAULT_RELATIVE_TO,
         };

         onPickerUpdate(getDatePickerValue({ from: { relative: relative.from }, to: { relative: relative.to } }));

         const defaultValue: IntervalValue = {
            from: {
               relative: relative.from,
               absolute: getAbsoluteDateFromRelative(relative.from),
            },
            to: {
               relative: relative.to,
               absolute: getAbsoluteDateFromRelative(relative.to),
            },
         };

         onUpdate(defaultValue, true);
      }
   }, [value, onUpdate]);

   const handlePickerUpdate = useCallback(
      (v: RangeDatepickerOutput) => {
         onPickerUpdate(v);

         const newFrom: IntervalItemValue = {
            absolute: parseDatepickerValue(v.from),
         };

         const newTo: IntervalItemValue = {
            absolute: parseDatepickerValue(v.to),
         };

         if (v.from.type === 'relative' && v.from.date) {
            newFrom.relative = v.from.date;
         }

         if (v.to.type === 'relative' && v.to.date) {
            newTo.relative = v.to.date;
         }

         onUpdate({ from: newFrom, to: newTo }, true);
      },
      [onUpdate],
   );

   // const { fromMillis, toMillis, onUpdateMillis } = useThinTimelineRuler({ from, to, onUpdate: handlePickerUpdate });

   return (
      <div className={classes.wrapper}>
         <ThinTimelineControls
            alwaysShowAsAbsolute={false}
            className={classes.controls}
            from={from}
            to={to}
            onUpdate={handlePickerUpdate}
            timeFormat={'HH:mm:ss'}
            toolbarPresets={['1h', '3h', '1d']}
            // relativeRangesPalette={['1h', '', '1d']} // TODO: DEPLOY-5980
            withNow={false}
            // TODO:
            // refresh не работает + кажется, что смысл в каком-то автообновлении есть только для свежих логов ("без дат")
            withRefresh={false}
            // defaultRefreshInterval={DEFAULT_REFRESH_INTERVAL}
            refreshInterval={refreshInterval} // DATAUI-1163
            onRefreshIntervalUpdate={setRefreshInterval} // DATAUI-1163
         />
         {/*  <ThinTimelineRuler
             from={fromMillis}
             to={toMillis}
             displayNow={true}
             hasZoomButtons={true}
             onUpdate={onUpdateMillis}
          /> */}
      </div>
   );
};

export const IntervalFilter = Object.assign(IntervalFilterImpl, {
   getValueFromUrlParams(urlParams: PlainObject): IntervalValue {
      const result: PlainObject = {};

      const { from, to } = urlParams;

      if (from) {
         const absolute = Number(from);

         result.from = absolute
            ? { absolute: new Date(absolute * 1000) }
            : {
                 absolute: getAbsoluteDateFromRelative(from),
                 relative: from,
              };
      }

      if (to) {
         const absolute = Number(to);

         result.to = absolute
            ? { absolute: new Date(absolute * 1000) }
            : {
                 absolute: getAbsoluteDateFromRelative(to),
                 relative: to,
              };
      }

      return result;
   },
   setUrlParamsFromValue(v: IntervalValue | undefined): PlainObject {
      const result: PlainObject = {};

      if (!v) {
         return result;
      }

      const { from, to } = v;

      if (from) {
         const { absolute, relative } = from;

         if (relative) {
            result.from = relative;
         } else if (absolute) {
            result.from = getDateUrlFormat(absolute!);
         }
      }

      if (to) {
         const { absolute, relative } = to;

         if (relative) {
            result.to = relative;
         } else if (absolute) {
            result.to = getDateUrlFormat(absolute!);
         }
      }

      return result;
   },
});
