import { HelpPopover, useTheme } from '@yandex-cloud/uikit';
import { classNames, modalService, Skeleton } from '@yandex-infracloud-ui/libs';
import { LazyMonacoEditor } from '@yandex-infracloud-ui/monaco-editor';
import type { editor as monacoEditor, IDisposable } from 'monaco-editor/esm/vs/editor/editor.api';
import * as React from 'react';
import { useCallback, useEffect, useRef, useState } from 'react';

import { RequestParams } from '../../../api/models/RequestParams';
import { useLogPageContext } from '../../../useLogPageContext';
import { LogQuerySyntaxHint } from '../LogQuerySyntaxHint/LogQuerySyntaxHint';

import { logQueryEditorOptions } from './logQueryEditorOptions';
import classes from './LogQueryInput.module.css';
import { registerLogQueryLanguage } from './registerLogQueryLanguage';

export interface QueryValue {
   query: string;
   request: RequestParams;
}

interface Props {
   className?: string;
   value: QueryValue;

   onUpdate(v: QueryValue): void;
}

let suggestTimer: ReturnType<typeof setTimeout>;

const onSetup = (editor: monacoEditor.IStandaloneCodeEditor): IDisposable => {
   const triggerSuggest = () => {
      suggestTimer = setTimeout(() => {
         if (!editor.hasTextFocus()) {
            return;
         }

         editor.trigger('suggest', 'editor.action.triggerSuggest', null);
      }, 100);
   };

   const disposables: IDisposable[] = [
      editor.onDidFocusEditorText(() => {
         triggerSuggest();
      }),

      editor.onDidChangeCursorSelection(({ selection }) => {
         if (selection.isEmpty()) {
            triggerSuggest();
         }
      }),
   ];

   return {
      dispose() {
         clearTimeout(suggestTimer);
         disposables.filter(Boolean).forEach(e => e.dispose());
      },
   };
};

export const LogQueryInput: React.FC<Props> = React.memo(({ className, value, onUpdate }) => {
   const [commonTheme] = useTheme();
   const [initiated, setInitiated] = useState(false);

   const { queryProvider } = useLogPageContext()!;
   const optionsRef = useRef<monacoEditor.IStandaloneEditorConstructionOptions>({});

   useEffect(() => {
      const { language, theme, dispose } = registerLogQueryLanguage(queryProvider);

      queryProvider.init(() => {
         optionsRef.current = {
            language,
            theme: commonTheme === 'dark' ? `${theme}-dark` : theme,
            ...logQueryEditorOptions,
         };

         setInitiated(true);
      });

      return dispose;
   }, [commonTheme, queryProvider]);

   const onQueryUpdate = useCallback(
      (q: string) => {
         onUpdate({
            query: q,

            /**
             * TODO: перенесла queryProvider.buildRequestParams() в BaseLogPage/filtersToRequestParams,
             * чтобы обрабатывать query не только из саджеста, но и из урла,
             * нужно ли здесь вообще хранить параметры для апи и зачем
             * (пока не буду, но возможно, стоит это отсюда оторвать?)
             */
            request: {},
         });
      },
      [onUpdate],
   );

   // useEffect(() => {
   //    window.setTimeout(() => {
   //       debugger;
   //    }, 10000);
   // }, []);
   const openHelp = useCallback(() => {
      modalService
         .openWithLayout(
            {
               // eslint-disable-next-line @typescript-eslint/no-empty-function
               onDismiss() {},
               showFooter: false,
               title: 'Query syntax help',
            },
            LogQuerySyntaxHint,
            { fields: queryProvider.fields },
         )
         .subscribe();

      return false;
   }, [queryProvider.fields]);

   return (
      <div className={classNames(classes.wrapper, className)}>
         <div className={classes.editorPlaceholder}>
            {initiated ? (
               <LazyMonacoEditor
                  onSetup={onSetup}
                  onUpdate={onQueryUpdate}
                  options={optionsRef.current}
                  // placeholder={'Search parameters...'} // не работает фокус по клику #INFRACLOUDUI-151
                  rows={2}
                  minRows={2}
                  isAutoRows={false}
                  value={value.query}
               />
            ) : (
               <Skeleton items={[{ width: '100%', size: 44, space: 0 }]} />
            )}
         </div>

         <p className={classes.hint}>
            Run search: <kbd>Ctrl</kbd>+<kbd>Enter</kbd>. Query syntax:{' '}
            <code>
               key1=value; key2="value with spaces, \"quotes\" and other punctuation characters (=,;!)"; key3=val1,
               val2; key4!=val1, val2
            </code>
            <HelpPopover openOnHover={false} onClick={openHelp} />
         </p>
      </div>
   );
});

LogQueryInput.displayName = 'LogQueryInput';
