import type { CancellationToken, IDisposable, Position } from 'monaco-editor';
import type { editor as editorTypes, languages as languagesTypes } from 'monaco-editor/esm/vs/editor/editor.api';
import { editor, languages } from 'monaco-editor/esm/vs/editor/editor.api';
import { firstValueFrom, map, Observable, takeUntil } from 'rxjs';

import type { QueryProvider } from '../../provider/QueryProvider';

const languageId = 'logQueryLanguage';

interface RegisterLanguageResult extends IDisposable {
   language: string;
   theme: string;
}

export function registerLogQueryLanguage(queryProvider: QueryProvider): RegisterLanguageResult {
   let disposables: IDisposable[] = [];

   const exist = languages.getLanguages().find(v => v.id === languageId);

   if (!exist) {
      languages.register({ id: languageId });
   }

   editor.defineTheme(languageId, {
      base: 'vs',
      inherit: true,
      rules: [
         { token: 'string', foreground: '000000d9' }, // var(--yc-color-text-primary
         { token: 'tag', foreground: 'a63eb2', fontStyle: 'bold' }, // var(--yc-color-text-utility)
         { token: 'operators', foreground: '8ea2b4' }, // var(--yc-color-text-misc)
         { token: 'delimiter', foreground: '8ea2b4' }, // var(--yc-color-text-misc)
      ],
      colors: {},
   });

   editor.defineTheme(`${languageId}-dark`, {
      base: 'vs-dark',
      inherit: true,
      rules: [
         { token: 'string', foreground: 'ffffffd9' }, // var(--yc-color-text-primary
         { token: 'tag', foreground: 'a63eb2', fontStyle: 'bold' }, // var(--yc-color-text-utility)
         { token: 'operators', foreground: '8ea2b4' }, // var(--yc-color-text-misc)
         { token: 'delimiter', foreground: '8ea2b4' }, // var(--yc-color-text-misc)
      ],
      colors: {},
   });

   disposables = [
      languages.setLanguageConfiguration(languageId, {
         autoClosingPairs: [{ open: '"', close: '"' }],
         surroundingPairs: [{ open: '"', close: '"' }],
      }),

      languages.setMonarchTokensProvider(languageId, {
         operators: ['=', '!=', '~', '!~'],
         ignoreCase: true,
         tokenizer: {
            root: [
               [/([^,";=~!]+)(\s*)(=|!=|~|!~)/, ['tag', '', 'operators']], // deploy context key can contain spaces
               [/[^,";\s=~!]+/, 'string'],
               [/"[^"]*"/, 'string'],
               [/[,;]/, 'delimiter'],
            ],
            whitespace: [[/\s*/, '']],
         },
      }),

      languages.registerCompletionItemProvider(languageId, {
         triggerCharacters: [' ', ',', ';', '=', '~'],
         provideCompletionItems(
            model: editorTypes.ITextModel,
            position: Position,
            context: languagesTypes.CompletionContext,
            token: CancellationToken,
         ): languagesTypes.ProviderResult<languagesTypes.CompletionList> {
            const cancelSubject = new Observable(observer => {
               const cancel = () => {
                  observer.next();
                  observer.complete();
               };

               if (token.isCancellationRequested) {
                  cancel();
               } else {
                  token.onCancellationRequested(cancel);
               }
            });

            return firstValueFrom(
               queryProvider.getSuggestions(model.getValue(), position).pipe(
                  map(suggestions => ({ suggestions })),
                  takeUntil(cancelSubject),
               ),
               { defaultValue: { suggestions: [] } },
            );
         },
      }),
   ];

   return {
      language: languageId,
      theme: languageId,
      dispose() {
         disposables.forEach(d => d.dispose());
         disposables.length = 0;
      },
   };
}
