import { Table, TableColumnConfig } from '@yandex-cloud/uikit';
import { SmartTab, SmartTabs } from '@yandex-infracloud-ui/libs';
import * as React from 'react';
import { useState } from 'react';

import { queryOperatorMap } from '../../../api/models/Operator';
import { QueryField } from '../../provider/models';

import classes from './LogQuerySyntaxHint.module.css';

/**
 * "IDE" for BNF is there https://bnfplayground.pauliankline.com/
 */
// language=TEXT
const grammar = `
<query> ::= <filter> "; " <filter> ";"?

<filter> ::= <key> <op> <values>

<key> ::= <text>

<op> ::= <eq_op> | <grep_op>
<eq_op> ::= "=" | "!="
<grep_op> ::= "~" | "!~"

<values> ::= <value> "," <value>
<value> ::= <s_value> | <q_value>
<s_value> ::= <text>
<q_value> ::= "\\"" <text_w_space> "\\""

<text>      ::= <character> <text> | <character>
<character> ::= <letter> | <digit>

<text_w_space> ::= <character_w_space> <text> | <character_w_space>
<character_w_space> ::= <letter> | <digit> | <space>

<letter>    ::= ([A-Z] [a-z])
<digit>     ::= [0-9]
<space>     ::= " "
`.trim();

const columns: TableColumnConfig<QueryField>[] = [
   {
      id: 'name',
   },
   {
      id: 'keyType',
   },
   {
      id: 'type',
   },
   {
      id: 'operators',
      template: item =>
         item.operators.map(o => (
            <code key={o} title={o}>
               {queryOperatorMap[o]}
            </code>
         )),
   },
];

enum QuerySyntaxTabs {
   Syntax = 'Syntax',
   Fields = 'Supported fields',
   Grammar = 'Grammar',
}

interface Props {
   fields: QueryField[];
}

export const LogQuerySyntaxHint: React.FC<Props> = ({ fields }) => {
   const [tab, setTab] = useState(QuerySyntaxTabs.Syntax);

   return (
      <article className={classes.wrapper}>
         <SmartTabs activeTab={tab} onSelectTab={setTab as (v: string) => void}>
            <SmartTab id={QuerySyntaxTabs.Syntax} title={QuerySyntaxTabs.Syntax}>
               <h4>Description</h4>
               <p>Query is a list of conditionals, translated to API request.</p>
               <p>Each conditional is an expression with operator and two operands.</p>

               <p>
                  The first operand is a field. It depends on API. See{' '}
                  {/* eslint-disable-next-line jsx-a11y/no-static-element-interactions,jsx-a11y/anchor-is-valid */}
                  <a onClick={() => setTab(QuerySyntaxTabs.Fields)}>"Supported fields"</a> tab for details.
               </p>
               <p>
                  Next part is one of the supported operators: <code>=</code>, <code>!=</code> for any fields and{' '}
                  <code>~</code> and <code>!~</code> for some string fields.
               </p>
               <p>
                  The last part is a comma separated values list. Each value with spaces should be quoted with{' '}
                  <code>"</code>.
               </p>
               {/* <h4>Examples</h4>
               <p>TODO</p> */}
            </SmartTab>

            <SmartTab id={QuerySyntaxTabs.Fields} title={QuerySyntaxTabs.Fields}>
               <Table className={classes.table} columns={columns} data={fields} />
            </SmartTab>

            <SmartTab id={QuerySyntaxTabs.Grammar} title={QuerySyntaxTabs.Grammar}>
               <h4>Grammar in BNF</h4>
               <pre>{grammar}</pre>
            </SmartTab>
         </SmartTabs>
      </article>
   );
};

LogQuerySyntaxHint.displayName = 'LogQuerySyntaxHint';
