import { Operator } from '../../api/models/Operator';

import { ExpressionNode, KeyNode, OperatorNode, QueryNode, ValueNode, ValuesNode } from './nodes';
import { BaseVisitor, parse } from './parser';

export interface QueryExpression {
   key: string;
   operator: Operator | null;
   values: string[];
}

class Visitor extends BaseVisitor {
   constructor() {
      super();

      this.validateVisitor();
   }

   public query(ctx: QueryNode['children']): QueryExpression[] {
      return ctx.expression?.map(e => this.visit(e)) ?? [];
   }

   public expression(ctx: ExpressionNode['children']): QueryExpression {
      return {
         key: this.visit(ctx.key),
         operator: this.visit(ctx.operator),
         values: this.visit(ctx.values),
      };
   }

   public key(ctx: KeyNode['children']): string {
      return ctx.Value[0].image;
   }

   public operator(ctx: OperatorNode['children']): Operator | null {
      if (ctx.Grep) {
         return Operator.Grep;
      }

      if (ctx.NotGrep) {
         return Operator.NotGrep;
      }

      if (ctx.NotEqual) {
         return Operator.NotEqual;
      }

      return Operator.Equal;
   }

   public values(ctx: ValuesNode['children']): string[] {
      return ctx.value?.map(v => this.visit(v)).filter(Boolean) ?? [];
   }

   public value(ctx: ValueNode['children']): string | null {
      if (ctx.Value) {
         return ctx.Value[0].image;
      }

      if (ctx.QuotedValue) {
         return ctx.QuotedValue[0].image.slice(1, -1);
      }

      return null;
   }
}

const visitor = new Visitor();

export function toAst(query: string): QueryExpression[] {
   return visitor.visit(parse(query));
}
