import { createSyntaxDiagramsCode, CstParser } from 'chevrotain';

import { ParseError } from './errors';
import { allTokens, Comma, Equal, Grep, NotEqual, NotGrep, QuotedValue, Separator, tokenize, Value } from './tokens';

class KeyValueParser extends CstParser {
   public key = this.RULE('key', () => {
      this.CONSUME(Value);
   });

   public operator = this.RULE('operator', () => {
      this.OR([
         { ALT: () => this.CONSUME(NotEqual) },
         { ALT: () => this.CONSUME(Equal) },
         { ALT: () => this.CONSUME(NotGrep) },
         { ALT: () => this.CONSUME(Grep) },
      ]);
   });

   public value = this.RULE('value', () => {
      this.OPTION(() => {
         this.OR([
            {
               ALT: () => this.CONSUME(Value),
            },
            {
               ALT: () => this.CONSUME(QuotedValue),
            },
         ]);
      });
   });

   public values = this.RULE('values', () => {
      this.MANY_SEP({
         SEP: Comma,
         DEF: () => {
            this.SUBRULE2(this.value);
         },
      });
   });

   public expression = this.RULE('expression', () => {
      this.SUBRULE(this.key);

      this.OPTION(() => {
         this.SUBRULE(this.operator);
         this.SUBRULE(this.values);
      });
   });

   public query = this.RULE('query', () => {
      this.MANY(() => {
         this.SUBRULE(this.expression);

         this.OPTION(() => {
            this.CONSUME(Separator);
         });
      });
   });

   constructor() {
      super(allTokens, { recoveryEnabled: true, nodeLocationTracking: 'full' });

      this.performSelfAnalysis();
   }
}

const parser = new KeyValueParser();

export const BaseVisitor = parser.getBaseCstVisitorConstructor();

export function parse(query: string) {
   const lexResult = tokenize(query);
   parser.input = lexResult.tokens;

   const cst = parser.query();

   if (parser.errors.length > 0) {
      throw new ParseError(parser.errors[0].message);
   }

   return cst;
}

export function getDiagramCode() {
   const serializedGrammar = parser.getSerializedGastProductions();
   return createSyntaxDiagramsCode(serializedGrammar);
}
