import { createToken, Lexer } from 'chevrotain';

import { LexerError } from './errors';

export const WhiteSpace = createToken({
   name: 'WhiteSpace',
   pattern: /\s+/,
   group: Lexer.SKIPPED,
});
export const Value = createToken({ name: 'Value', pattern: /[^,";\s=~!]+/ });
export const QuotedValue = createToken({ name: 'QuotedValue', pattern: /"[^"]*"/ });
export const NotEqual = createToken({ name: 'NotEqual', pattern: /!=/, label: '!=' });
export const Equal = createToken({ name: 'Equal', pattern: /=/, label: '=' });
export const NotGrep = createToken({ name: 'NotGrep', pattern: /!~/, label: '!~' });
export const Grep = createToken({ name: 'Grep', pattern: /~/, label: '~' });
export const Comma = createToken({ name: 'Comma', pattern: /,/, label: ',' });
export const Separator = createToken({ name: 'ExpressionSeparator', pattern: /;/, label: ';' });

export const valueTokens = [Value, QuotedValue];
export const operatorTokens = [NotEqual, Equal, NotGrep, Grep];
export const allTokens = [WhiteSpace, ...operatorTokens, ...valueTokens, Comma, Separator];

const lexer = new Lexer(allTokens, { positionTracking: 'full' });

export function tokenize(text: string) {
   const result = lexer.tokenize(text);

   if (result.errors.length > 0) {
      throw new LexerError(result.errors[0].message);
   }

   return result;
}
