import { ApiHooks } from '@yandex-infracloud-ui/libs/@types/http/BaseApi';
import { of } from 'rxjs';

import { LogApi } from '../../api/LogApi';
import { Operator } from '../../api/models/Operator';
import { AWACS_LOG_ENDPOINT, DEPLOY_LOG_ENDPOINT } from '../../constants';

import { QueryProvider } from './QueryProvider';
import awacsSchema from './testData/awacsSchema.json';
import deploySchema from './testData/deploySchema.json';

const MOCK_API = true;

const hooks: ApiHooks = {
   before(r) {
      console.log('It is a real request:', r);
   },
};

describe('QueryProvider', () => {
   let deployTestApi: LogApi;
   let awacsTestApi: LogApi;

   beforeEach(() => {
      deployTestApi = new LogApi(DEPLOY_LOG_ENDPOINT, hooks);
      awacsTestApi = new LogApi(AWACS_LOG_ENDPOINT, hooks);

      if (MOCK_API) {
         deployTestApi.getSchema = jest.fn(() => of(deploySchema as any));
         awacsTestApi.getSchema = jest.fn(() => of(awacsSchema as any));
      }
   });

   describe('fill fields', () => {
      it('for AWACS', done => {
         const provider = new QueryProvider(awacsTestApi);
         provider.init(() => {
            console.table(provider.fields);
            done();
         });
      });

      it('for Deploy', done => {
         const provider = new QueryProvider(deployTestApi);
         provider.init(() => {
            console.table(provider.fields);
            done();
         });
      });
   });

   describe('public methods', () => {
      let provider: QueryProvider;
      beforeEach(done => {
         provider = new QueryProvider(awacsTestApi);
         provider.init(() => done());
      });

      it('should suggest keys', done => {
         provider.suggestKeys(new Set(), 'client').subscribe(r => {
            expect(r).toEqual(['clientIp', 'clientPort']);
            done();
         });
      });

      function testValueSuggest(field: string, valuePrefix: string, expected: string[], done: jest.DoneCallback) {
         if (MOCK_API) {
            awacsTestApi.getSearchQuerySuggests = jest.fn(() => of(expected));
         }

         provider
            .suggestValues(
               [
                  {
                     key: 'namespace',
                     operator: Operator.Equal,
                     values: ['awacs'],
                  },
                  {
                     key: 'limit',
                     operator: Operator.Equal,
                     values: ['50'],
                  },
               ],
               field,
               valuePrefix,
            )
            .subscribe(r => {
               expect(r).toEqual(expected);
               done();
            });
      }

      it('should suggest METHOD with empty query', done => {
         testValueSuggest('method', '', ['POST', 'GET'], done);
      });

      it('should suggest METHOD with non-empty query', done => {
         testValueSuggest('method', 'g', ['GET'], done);
      });

      it('should suggest DOMAIN_', done => {
         testValueSuggest('domain', '', ['awacs.yandex-team.ru'], done);
      });
   });

   describe('getSuggestions', () => {
      let provider: QueryProvider;
      let suggestKeysMock: jest.Mock;
      let suggestValuesMock: jest.Mock;

      beforeEach(() => {
         provider = new QueryProvider(deployTestApi);

         suggestKeysMock = jest.fn(() => of([]));
         provider.suggestKeys = suggestKeysMock;

         suggestValuesMock = jest.fn(() => of([]));
         provider.suggestValues = suggestValuesMock;
      });

      function test(query: string) {
         return provider.getSuggestions(query.replace('|', ''), {
            column: query.indexOf('|') + 1,
            lineNumber: 1,
         });
      }

      it('suggestKeys for empty', () => {
         test('|');

         expect(suggestKeysMock.mock.calls.length).toBe(1);
         expect(suggestValuesMock.mock.calls.length).toBe(0);

         const [expressions, query] = suggestKeysMock.mock.calls[0];
         expect(expressions).toMatchInlineSnapshot(`Set {}`);
         expect(query).toBe('');
      });

      it('suggestKeys after key part', () => {
         test('meth|');

         expect(suggestKeysMock.mock.calls.length).toBe(1);
         expect(suggestValuesMock.mock.calls.length).toBe(0);

         const [expressions, query] = suggestKeysMock.mock.calls[0];
         expect(expressions).toMatchInlineSnapshot(`Set {}`);
         expect(query).toBe('meth');
      });

      it('suggestValues after operator', () => {
         test('method=|');

         expect(suggestKeysMock.mock.calls.length).toBe(0);
         expect(suggestValuesMock.mock.calls.length).toBe(1);

         const [expressions, fieldName, query] = suggestValuesMock.mock.calls[0];
         expect(expressions).toMatchInlineSnapshot(`Array []`); // TODO check
         expect(fieldName).toBe('method');
         expect(query).toBe('');
      });
   });
});
