import { camelCase, deepClone, hasIncluded, isEmpty, isOneOfErrors } from '@yandex-infracloud-ui/libs';
import type { JSONSchema7 } from 'json-schema';
import type { IPosition } from 'monaco-editor';
import type { languages } from 'monaco-editor/esm/vs/editor/editor.api';
import { map, Observable, of, tap } from 'rxjs';

import { PlainObject } from '../../../smart-table';
import { LogApi } from '../../api/LogApi';
import { ArrayRequestParam } from '../../api/models/ArrayRequestParam';
import { RequestParams } from '../../api/models/RequestParams';
import { safeParseFloat } from '../../helpers/safeParseFloat';
import { ProcessStatus } from '../../models';
import { QueryExpression, toAst } from '../grammar/ast';
import { LexerError, ParseError } from '../grammar/errors';

import { DifferentOperatorsError } from './errors';
import {
   cleanupExpressions,
   extractSchema,
   getFieldOperations,
   getFieldType,
   getOpType,
   getRequestPropertyNameByKeyType,
   getSelectType,
   normalizeEnums,
} from './helpers';
import { FieldType, QueryField, SuggestCase, SuggestRequest } from './models';
import { getSuggestRequest } from './suggestVisitor';

type CompletionItem = languages.CompletionItem;
const CompletionItemKindText = 18; // HARDCODE to prevent load monaco synchronously (enum is a value, not only the type)

interface GetRequestParamsArguments {
   addForcedParams: boolean;
   currentKey?: string;
   expressions: QueryExpression[];
}

interface GetRequestParamsFromFiltersArguments {
   addForcedParams: boolean;
   filters: PlainObject;
}

export class QueryProvider {
   public fields: QueryField[] = [];

   private currentRequestParams: RequestParams = {};

   private initHandlers: ((success: boolean) => void)[] = [];

   private status: ProcessStatus = ProcessStatus.New;

   constructor(
      private readonly api: LogApi,
      private readonly forcedRequestParams: RequestParams = {},
      private readonly requestFields?: QueryField[],
      private readonly customRequestFields?: QueryField[],
   ) {
      this.fields = this.requestFields ?? [];
   }

   public init(cb?: (success: boolean) => void) {
      if (cb) {
         this.initHandlers.push(cb);
      }

      if (this.status !== ProcessStatus.New) {
         const isInitiated = this.status === ProcessStatus.Success || this.status === ProcessStatus.Error;
         if (cb && isInitiated) {
            cb(this.status === ProcessStatus.Success);
         }

         return;
      }

      this.status = ProcessStatus.InProgress;

      // FIXME: при первой загрузке страницы не успевают подъехать поля из api, нужно починить
      // пока прокидываю готовый конфиг полей (т.к. сейчас api всё равно поломано #DEPLOY-5599)
      if (isEmpty(this.fields)) {
         this.api.getSchema().subscribe({
            next: resp => {
               // values
               const valuesSuggestSchema = extractSchema('GetSearchQuerySuggests', resp);
               const props = valuesSuggestSchema.properties!;
               const keyTypeProp = props.keyType as JSONSchema7;
               const keyTypes = normalizeEnums(keyTypeProp.enum as string[]);

               const requestSchema = props.request as JSONSchema7;

               for (const keyType of keyTypes) {
                  this.fields.push({
                     keyType,
                     name: camelCase(keyType), // FIXME: SNAKE_CASE support
                     operators: getFieldOperations(requestSchema, keyType),
                     type: getFieldType(requestSchema, keyType),
                  });
               }

               // object keys
               // const objectKeysSchema = extractSchema('GetQueryContextKeys', resp);

               this.initHandlers.forEach(h => h(true));
               this.status = ProcessStatus.Success;
            },
            error: () => {
               this.initHandlers.forEach(h => h(false));
               this.status = ProcessStatus.Error;
            },
         });
      } else {
         this.initHandlers.forEach(h => h(true));
         this.status = ProcessStatus.Success;
      }
   }

   /**
    * Устанавливает текущие параметры запроса, которые будут добавлены в запрос в suggestValues...
    */
   public setCurrentRequestParams(v: RequestParams) {
      this.currentRequestParams = v;
   }

   /**
    * Для деплоя (Deploy Unit Id)
    */
   public getForcedRequestParams() {
      return deepClone(this.forcedRequestParams);
   }

   /**
    * Вызывается из редактора, должен вернуть асинхронно пункта саджеста
    *
    * Под капотом дергает suggestKeys и suggestValues
    */
   public getSuggestions(value: string, position: IPosition): Observable<CompletionItem[]> {
      let expressions: QueryExpression[];
      try {
         expressions = cleanupExpressions(toAst(value));
      } catch (e) {
         if (isOneOfErrors([DifferentOperatorsError, ParseError, LexerError], e)) {
            console.warn('Invalid query', value, e);
            return of([]);
         }

         throw e;
      }

      // ParseError and LexerError has already excluded
      const suggestRequest = getSuggestRequest(value, position);

      const devLog = tap((suggestions: CompletionItem[]) => {
         console.log({ position, suggestRequest, suggestions });
      });

      switch (suggestRequest.type) {
         case 'key': {
            const usedKeys = new Set(cleanupExpressions(expressions).map(e => e.key)); // TODO handle exceptions

            return this.suggestKeys(usedKeys, suggestRequest.query).pipe(
               this.mapToKeysSuggestions(suggestRequest),
               devLog,
            );
         }
         case 'value': {
            return this.suggestValues(
               cleanupExpressions(expressions),
               suggestRequest.fieldName!,
               suggestRequest.query,
            ).pipe(this.mapToValuesSuggestions(suggestRequest), devLog);
         }
         default: {
            console.warn('No suggestions');

            return of([]);
         }
      }
   }

   /**
    * Запрашивает возможные ключи (фильтры)
    */
   public suggestKeys(usedKeys: Set<string>, q: string): Observable<string[]> {
      return of(
         this.fields
            .filter(f => !usedKeys.has(f.name)) // Пропуск уже использованных
            .filter(f => hasIncluded(q, f.keyType, f.name)) // Фильтр по вхождению подстроки
            .map(f => f.name),
      );
   }

   /**
    * Запрашивает возможные значения для текущего фильтра (из API)
    */
   public suggestValues(expressions: QueryExpression[], fieldName: string, q: string): Observable<string[]> {
      const field = this.fields.find(f => f.name === fieldName);
      if (!field) {
         console.warn(`Unknown field ${fieldName}`);

         return of([]);
      }

      return this.api.getSearchQuerySuggests({
         valuePrefix: q,
         keyType: field.keyType,
         request: this.buildRequestParams({
            addForcedParams: true,
            currentKey: field.keyType,
            expressions: expressions.filter(e => e.key !== field.name),
         }),
      });
   }

   /**
    * Конвертирует выражения из запроса в набор запросов к API (фильтры)
    *
    * Также подставляет в запрос переданные "жестко" фильтры (например, namespace в AWACS или stageId в Deploy)
    */
   public buildRequestParams({ addForcedParams, expressions, currentKey }: GetRequestParamsArguments): RequestParams {
      const paramsFromExpressions = expressions.reduce((acc, p) => {
         const [key, ...path] = p.key.split('.'); // deploy context

         // customRequestFields - дополнительные поля, которые используем только для поиска (по ним не саджестим и не кликаем)
         const field = this.customRequestFields?.find(f => f.name === key) ?? this.fields.find(f => f.name === key);

         // direct param
         if (!field) {
            const [firstValue] = p.values;
            acc[key] = firstValue;

            return acc;
         }

         // list param
         const arrayRequestParam: ArrayRequestParam = {
            values: field.type === FieldType.Number ? p.values.map(v => safeParseFloat(v)) : p.values,
         };

         if (p.operator) {
            // Skip default selectType
            const selectType = getSelectType(p.operator);
            if (selectType !== 'INCLUDE') {
               arrayRequestParam.selectType = selectType;
            }

            // Skip default opType
            const opType = getOpType(p.operator);
            if (opType !== 'EQ') {
               arrayRequestParam.opType = opType;
            }
         }

         const requestPropertyName = getRequestPropertyNameByKeyType(key);

         // deploy context.*** отправляем в другом формате
         if (!isEmpty(path)) {
            arrayRequestParam.path = path.join('.');

            if (!acc[requestPropertyName]?.values) {
               acc[requestPropertyName] = { values: [] };
            }

            acc[requestPropertyName].values.push(arrayRequestParam);
         } else {
            acc[requestPropertyName] = arrayRequestParam;
         }

         return acc;
      }, {} as RequestParams);

      const params = this.addParams(addForcedParams, paramsFromExpressions);
      if (currentKey) {
         delete params[getRequestPropertyNameByKeyType(currentKey)];
      }

      return params;
   }

   public buildRequestParamsFromFilters({
      addForcedParams,
      filters,
   }: GetRequestParamsFromFiltersArguments): RequestParams {
      const params = Object.entries(filters).reduce((acc, [name, value]) => {
         const field = this.fields.find(f => f.name === name);

         // direct param
         if (!field) {
            acc[name] = value;
            return acc;
         }

         // empty filter
         if (!value || isEmpty(value)) {
            return acc;
         }

         // list param
         acc[getRequestPropertyNameByKeyType(field.keyType)] = {
            // используется для Deploy LevelFilter (string[])
            values: field.type === FieldType.Number ? value.map((v: any) => safeParseFloat(v)) : value,
         };

         // INCLUDE/EXCLUDE and EQ/GREP skipped

         return acc;
      }, {} as RequestParams);

      return this.addParams(addForcedParams, params);
   }

   /**
    * Добавляет в переданные параметры форсированные (заданные при инициализации) и дополнительные (текущие)
    */
   private addParams(addForcedParams: boolean, params: RequestParams): RequestParams {
      if (!addForcedParams) {
         return params;
      }

      return {
         ...this.forcedRequestParams,
         ...this.currentRequestParams,
         ...params,
      };
   }

   private mapToKeysSuggestions(config: SuggestRequest) {
      return map((candidates: string[]) =>
         candidates.map(c => {
            const insertText = `${c}=`;

            return {
               // detail: config.case, // debug field
               insertText,
               kind: CompletionItemKindText,
               label: c,
               range: {
                  ...config.range,
                  endColumn: config.range.startColumn + insertText.length,
               },
            } as CompletionItem;
         }),
      );
   }

   private mapToValuesSuggestions(config: SuggestRequest) {
      return map((candidates: string[]) =>
         candidates.map(c => {
            const insertText = config.case === SuggestCase.InQuotedValue ? c : `${c}; `;

            return {
               // detail: config.case, // debug field
               insertText,
               kind: CompletionItemKindText,
               label: c,
               range: {
                  ...config.range,
                  endColumn: config.range.startColumn + insertText.length,
               },
            } as CompletionItem;
         }),
      );
   }
}
