import { camelCase, deepClone, isEmpty } from '@yandex-infracloud-ui/libs';
import { JSONSchema7 } from 'json-schema';

import { EndpointSchema } from '../../api/models/EndpointSchema';
import { Operator } from '../../api/models/Operator';
import { OpType } from '../../api/models/OpType';
import { SelectType } from '../../api/models/SelectType';
import { QueryExpression, toAst } from '../grammar/ast';

import { DifferentOperatorsError } from './errors';
import { FieldType } from './models';

export function getSelectType(o: Operator): SelectType {
   return o.startsWith('Not') ? 'EXCLUDE' : 'INCLUDE';
}

export function getOpType(o: Operator): OpType {
   return o.includes('Grep') ? 'GREP' : 'EQ';
}

// Исправляем несоответствия с беком (видимо, ошибка) TODO поправить на беке!
// Ключ - keyType, значение - поле в request
const forcedMappings = new Map([['CONTAINER_ID', 'containerList']]);

export function extractSchema(name: string, resp: EndpointSchema[]): JSONSchema7 {
   for (const [epName, requestSchema] of resp) {
      if (epName === name) {
         return requestSchema;
      }
   }

   throw new Error(`Could not find ${name} in schemas`);
}

export function getRequestPropertyNameByKeyType(keyType: string): string {
   return forcedMappings.get(keyType) ?? `${camelCase(keyType)}List`;
}

function getRequestPropertyByKeyType(requestSchema: JSONSchema7, keyType: string) {
   const requestProperty = getRequestPropertyNameByKeyType(keyType);

   return requestSchema.properties![requestProperty] as JSONSchema7 | undefined;
}

export function getFieldOperations(requestSchema: JSONSchema7, keyType: string): Operator[] {
   const requestProperty = getRequestPropertyByKeyType(requestSchema, keyType);
   if (!requestProperty) {
      throw new Error(`Unknown request property for ${keyType}`);
   }

   const result: Operator[] = [Operator.Equal, Operator.NotEqual];
   const opTypeEnum = (requestProperty.properties!.opType as JSONSchema7)?.enum;
   if (opTypeEnum?.includes('GREP')) {
      result.push(Operator.Grep, Operator.NotGrep);
   }
   return result;
}

export function getFieldType(requestSchema: JSONSchema7, keyType: string): FieldType {
   const requestProperty = getRequestPropertyByKeyType(requestSchema, keyType);
   if (!requestProperty) {
      throw new Error(`Unknown request property for ${keyType}`);
   }

   const valuesType = ((requestProperty.properties!.values as JSONSchema7).items as JSONSchema7)?.type;

   return valuesType === 'string' ? FieldType.String : FieldType.Number;
}

/**
 * Normalize expressions
 *
 * * Remove expressions without value
 * * Merge multiple entries with the same operator
 */
export function cleanupExpressions(expressions: QueryExpression[]): QueryExpression[] {
   const expressionsWithValues = expressions.filter(e => e.operator && !isEmpty(e.values));
   const merged = new Map<string, QueryExpression>();

   for (const expression of expressionsWithValues) {
      const exist = merged.get(expression.key);
      if (exist) {
         if (exist.operator !== expression.operator) {
            throw new DifferentOperatorsError(expression.key);
         }
         exist.values.push(...expression.values);
      } else {
         merged.set(expression.key, deepClone(expression));
      }
   }

   return Array.from(merged.values());
}

export function getCleanExpressions(query: string): QueryExpression[] {
   let cleanExpressions: QueryExpression[] = [];

   try {
      cleanExpressions = cleanupExpressions(toAst(query));
   } catch (e) {
      console.warn(e);
   }

   return cleanExpressions;
}

/**
 * UNKNOWN - специальное зарезервированное первое значения для enum в protobuf
 *
 * По сути это null, поэтому стоит его рассматривать как отсутствие значения
 */
export function normalizeEnums(values: string[]) {
   return values.filter(v => v !== 'UNKNOWN');
}
