import { CstNode, CstNodeLocation, IToken, TokenType } from 'chevrotain';
import type { IRange } from 'monaco-editor';

import {
   isInsideLocation,
   isInsideOrRightAfterLocation,
   isRightAfterLocation,
   locationToRange,
   positionToRange,
   reduceToCurrent,
} from './locationHelpers';

describe('log-page/query/locationHelpers', () => {
   describe('isInsideLocation', () => {
      it('same line', () => {
         expect(
            isInsideLocation(
               { startLine: 1, startColumn: 10, endLine: 1, endColumn: 20 },
               { lineNumber: 1, column: 15 },
            ),
         ).toBe(true);

         expect(
            isInsideLocation(
               { startLine: 1, startColumn: 10, endLine: 1, endColumn: 20 },
               { lineNumber: 1, column: 25 },
            ),
         ).toBe(false);
      });

      it('multiline', () => {
         expect(
            isInsideLocation(
               { startLine: 1, startColumn: 20, endLine: 3, endColumn: 1 },
               { lineNumber: 2, column: 15 },
            ),
         ).toBe(true);

         expect(
            isInsideLocation({ startLine: 1, startColumn: 20, endLine: 3, endColumn: 1 }, { lineNumber: 3, column: 5 }),
         ).toBe(false);

         expect(
            isInsideLocation({ startLine: 1, startColumn: 20, endLine: 3, endColumn: 1 }, { lineNumber: 1, column: 5 }),
         ).toBe(false);
      });
   });

   describe('isRightAfterLocation', () => {
      it('same line', () => {
         expect(
            isRightAfterLocation(
               { startLine: 1, startColumn: 10, endLine: 1, endColumn: 20 },
               { lineNumber: 1, column: 21 },
            ),
         ).toBe(true);

         expect(
            isRightAfterLocation(
               { startLine: 1, startColumn: 10, endLine: 1, endColumn: 20 },
               { lineNumber: 1, column: 22 },
            ),
         ).toBe(false);
      });

      it('multiline', () => {
         expect(
            isRightAfterLocation(
               { startLine: 1, startColumn: 20, endLine: 3, endColumn: 1 },
               { lineNumber: 3, column: 2 },
            ),
         ).toBe(true);

         expect(
            isRightAfterLocation(
               { startLine: 1, startColumn: 20, endLine: 3, endColumn: 1 },
               { lineNumber: 3, column: 3 },
            ),
         ).toBe(false);

         expect(
            isRightAfterLocation(
               { startLine: 1, startColumn: 20, endLine: 3, endColumn: 1 },
               { lineNumber: 1, column: 5 },
            ),
         ).toBe(false);

         expect(
            isRightAfterLocation(
               { startLine: 1, startColumn: 20, endLine: 3, endColumn: 1 },
               { lineNumber: 2, column: 5 },
            ),
         ).toBe(false);
      });
   });

   describe('isInsideOrRightAfterLocation', () => {
      it('same line', () => {
         expect(
            isInsideOrRightAfterLocation(
               { startLine: 1, startColumn: 1, endLine: 1, endColumn: 10 },
               { lineNumber: 1, column: 6 },
            ),
         ).toBe(true);
         expect(
            isInsideOrRightAfterLocation(
               { startLine: 1, startColumn: 1, endLine: 1, endColumn: 10 },
               { lineNumber: 1, column: 11 },
            ),
         ).toBe(true);

         expect(
            isInsideOrRightAfterLocation(
               { startLine: 1, startColumn: 1, endLine: 1, endColumn: 10 },
               { lineNumber: 1, column: 12 },
            ),
         ).toBe(false);
      });

      it('multiline', () => {
         expect(
            isInsideOrRightAfterLocation(
               { startLine: 1, startColumn: 1, endLine: 3, endColumn: 10 },
               { lineNumber: 3, column: 5 },
            ),
         ).toBe(true);

         expect(
            isInsideOrRightAfterLocation(
               { startLine: 1, startColumn: 1, endLine: 3, endColumn: 10 },
               { lineNumber: 3, column: 11 },
            ),
         ).toBe(true);

         expect(
            isInsideOrRightAfterLocation(
               { startLine: 1, startColumn: 1, endLine: 3, endColumn: 10 },
               { lineNumber: 2, column: 11 },
            ),
         ).toBe(true);

         expect(
            isInsideOrRightAfterLocation(
               { startLine: 1, startColumn: 1, endLine: 3, endColumn: 10 },
               { lineNumber: 5, column: 11 },
            ),
         ).toBe(false);
      });
   });

   describe('reduceToCurrent', () => {
      const [l1, l2, l3]: CstNodeLocation[] = [
         { startLine: 1, startColumn: 1, endLine: 1, endColumn: 5, startOffset: 1, endOffset: 6 },
         { startLine: 1, startColumn: 6, endLine: 1, endColumn: 8, startOffset: 6, endOffset: 9 },
         { startLine: 1, startColumn: 9, endLine: 1, endColumn: 11, startOffset: 6, endOffset: 12 },
      ];

      it('should work for nodes', () => {
         const nodes = [{ location: l1 } as CstNode, { location: l2 } as CstNode, { location: l3 } as CstNode];

         expect(nodes.reduce(reduceToCurrent({ lineNumber: 1, column: 7 }))).toBe(nodes[1]);
      });

      it('should work for tokens', () => {
         const tokens = [
            { tokenType: {} as TokenType, ...l1 } as IToken,
            { tokenType: {} as TokenType, ...l2 } as IToken,
            { tokenType: {} as TokenType, ...l3 } as IToken,
         ];

         expect(tokens.reduce(reduceToCurrent({ lineNumber: 1, column: 7 }))).toBe(tokens[1]);
      });
   });

   it('locationToRange', () => {
      expect(locationToRange({ startLine: 1, startColumn: 20, endLine: 3, endColumn: 1 })).toEqual({
         startLineNumber: 1,
         startColumn: 20,
         endLineNumber: 3,
         endColumn: 1,
      } as IRange);
   });

   it('locationToRange', () => {
      expect(positionToRange({ lineNumber: 1, column: 20 })).toEqual({
         startLineNumber: 1,
         startColumn: 20,
         endLineNumber: 1,
         endColumn: 20,
      } as IRange);
   });
});
