import { CstElement, CstNode, CstNodeLocation, IToken } from 'chevrotain';
import type { IPosition, IRange } from 'monaco-editor';

export type Location = Required<Omit<CstNodeLocation, 'startOffset' | 'endOffset'>>;

export function isInsideLocation(l: Location, p: IPosition) {
   // Всё на одной строке (самый частый случай)
   if (l.startLine === l.endLine && l.startLine === p.lineNumber) {
      return p.column > l.startColumn && p.column < l.endColumn;
   }

   // Первая строка многострочного узла
   if (l.startLine === p.lineNumber) {
      return p.column > l.startColumn;
   }

   // Последняя строка многострочного узла
   if (l.endLine === p.lineNumber) {
      return p.column < l.endColumn;
   }

   // Между строк
   return p.lineNumber > l.startLine && p.lineNumber < l.endLine;
}

export function isRightAfterLocation(l: Location, p: IPosition) {
   // Всё на одной строке (самый частый случай)
   if (l.startLine === l.endLine && l.startLine === p.lineNumber) {
      return p.column === l.endColumn + 1;
   }

   // Первая строка многострочного узла
   if (l.startLine === p.lineNumber) {
      return false;
   }

   // Последняя строка многострочного узла
   if (l.endLine === p.lineNumber) {
      return p.column === l.endColumn + 1;
   }

   // Между строк
   return false;
}

export function isInsideOrRightAfterLocation(l: Location, p: IPosition) {
   return isInsideLocation(l, p) || isRightAfterLocation(l, p);
}

export function reduceToCurrent<T extends CstElement>(p: IPosition) {
   function findCurrentToken(a: IToken, b: IToken) {
      return isInsideOrRightAfterLocation(b as Location, p) ? b : a;
   }

   function findCurrentNode(a: CstNode, b: CstNode) {
      return isInsideOrRightAfterLocation(b.location as Location, p) ? b : a;
   }

   return (a: T, b: T): T => {
      const isToken = Boolean((a as IToken).tokenType);

      return isToken
         ? (findCurrentToken(a as IToken, b as IToken) as T)
         : (findCurrentNode(a as CstNode, b as CstNode) as T);
   };
}

export function locationToRange(l: Location): IRange {
   return {
      startLineNumber: l.startLine,
      startColumn: l.startColumn,
      endLineNumber: l.endLine,
      endColumn: l.endColumn,
   };
}

export function positionToRange(p: IPosition) {
   return {
      startLineNumber: p.lineNumber,
      startColumn: p.column,
      endLineNumber: p.lineNumber,
      endColumn: p.column,
   };
}
